package webpms

import (
	"context"
	"fmt"
	"io"
	"net/url"
	"time"

	"github.com/go-resty/resty/v2"

	"a.yandex-team.ru/library/go/core/xerrors"
)

type Client struct {
	config     Config
	httpClient *resty.Client
}

type findBookingsQueryParam struct {
	name  string
	value string
}

type FindBookingsFilter []findBookingsQueryParam

const xAPIHeaderName = "X-API-KEY"

func NewClient(cfg Config) *Client {
	return &Client{
		config:     cfg,
		httpClient: resty.New().SetTimeout(cfg.Timeout),
	}
}

func (c *Client) GetBooking(ctx context.Context, number string) (*BookingDTO, error) {
	endpoint := fmt.Sprintf("/bookings/%s", url.PathEscape(number))
	var result BookingDTO
	if err := c.doExecute(ctx, resty.MethodGet, c.getURL(endpoint), nil, &result, url.Values{}); err != nil {
		return nil, xerrors.Errorf("unable to get booking with number=%s: %w", number, err)
	}
	return &result, nil
}

func (c *Client) FindBookings(ctx context.Context, filters ...FindBookingsFilter) (*FindBookingsResponse, error) {
	query := url.Values{}
	for _, filter := range filters {
		for _, param := range filter {
			query.Add(param.name, param.value)
		}
	}
	var result FindBookingsResponse
	if err := c.doExecute(ctx, resty.MethodGet, c.getURL("/bookings"), nil, &result, query); err != nil {
		return nil, xerrors.Errorf("unable to find bookings: %w", err)
	}
	return &result, nil
}

func (c *Client) ListRooms(ctx context.Context) (*ListRoomsResponse, error) {
	query := url.Values{}
	var result ListRoomsResponse
	if err := c.doExecute(ctx, resty.MethodGet, c.getURL("/rooms"), nil, &result, query); err != nil {
		return nil, xerrors.Errorf("unable to list rooms: %w", err)
	}
	return &result, nil
}

func (c *Client) getURL(endpoint string) string {
	return fmt.Sprintf("%s%s", c.config.BaseURL, endpoint)
}

func (c *Client) doExecute(ctx context.Context, method, path string, body, result interface{}, queryParams url.Values) error {
	var errResponse map[string]interface{}
	r := c.httpClient.R().
		SetContext(ctx).
		SetHeader(xAPIHeaderName, c.config.APIKey).
		SetBody(body).
		SetError(&errResponse)
	if result != nil {
		r = r.SetResult(result)
	}
	if queryParams != nil {
		r = r.SetQueryParamsFromValues(queryParams)
	}
	response, err := r.Execute(method, path)
	if err != nil {
		return ResponseError.Wrap(err)
	}
	if !response.IsSuccess() {
		raw, _ := io.ReadAll(response.RawResponse.Body)
		return xerrors.Errorf("unexpected response from tl web pms service: %v", StatusError{
			Status:      response.StatusCode(),
			Response:    errResponse,
			ResponseRaw: string(raw),
		})
	}
	return nil
}

func Modified(from time.Time, to time.Time) FindBookingsFilter {
	return FindBookingsFilter{
		{
			name:  "findBookingsDto.modifiedFrom",
			value: from.Format(hotelTimeLayout),
		},
		{
			name:  "findBookingsDto.modifiedTo",
			value: to.Format(hotelTimeLayout),
		},
	}
}

func BookingLookupState(state bookingLookupState) FindBookingsFilter {
	return FindBookingsFilter{
		{
			name:  "findBookingsDto.state",
			value: string(state),
		},
	}
}

func AffectsPeriod(from time.Time, to time.Time) FindBookingsFilter {
	return FindBookingsFilter{
		{
			name:  "findBookingsDto.affectsPeriodFrom",
			value: from.Format(hotelTimeLayout),
		},
		{
			name:  "findBookingsDto.affectsPeriodTo",
			value: to.Format(hotelTimeLayout),
		},
	}
}
