package webpms

import (
	"fmt"
	"strings"
	"time"
)

const hotelTimeLayout = "2006-01-02T15:04"
const dateLayout = "2006-01-02"

type bookingLookupState string
type RoomStayStatus string
type BookingStatus string

const (
	BookingLookupStateActive    bookingLookupState = "Active"
	BookingLookupStateCancelled bookingLookupState = "Cancelled"
)

const (
	RoomStayStatusNew        RoomStayStatus = "Now"
	RoomStayStatusCheckedIn  RoomStayStatus = "CheckedIn"
	RoomStayStatusCheckedOut RoomStayStatus = "CheckedOut"
	RoomStayStatusCancelled  RoomStayStatus = "Cancelled"
)

const (
	BookingStatusUnconfirmed BookingStatus = "Unconfirmed"
	BookingStatusConfirmed   BookingStatus = "Confirmed"
	BookingStatusCancelled   BookingStatus = "Cancelled"
	BookingStatusReleased    BookingStatus = "Released"
	BookingStatusPending     BookingStatus = "Pending"
	BookingStatusEdited      BookingStatus = "Edited"
)

type Timestamp struct {
	time.Time
}

type HotelTime struct {
	time.Time
}

type Date struct {
	time.Time
}

type FindBookingsResponse struct {
	BookingNumbers []string `json:"bookingNumbers"`
}

type ListRoomsResponse []RoomDTO

type DictionaryItemDTO struct {
	Key   string `json:"key"`
	Value string `json:"value"`
}

type GuestDTO struct {
	ID              string             `json:"id"`
	LastName        string             `json:"lastName"`
	FirstName       string             `json:"firstName"`
	MiddleName      string             `json:"middleName"`
	BirthDate       *Date              `json:"birthDate"`
	CitizenshipCode string             `json:"citizenshipCode"`
	Status          *DictionaryItemDTO `json:"status"`
	Emails          []string           `json:"emails"`
	Phones          []string           `json:"phones"`
	Gender          string             `json:"gender"`
}

type CompanyDTO struct {
	ID   string `json:"id"`
	Name string `json:"name"`
}

type GuestCountInfoDTO struct {
	Adults   int `json:"adults"`
	Children int `json:"children"`
}

type AmountDTO struct {
	Amount   float64 `json:"amount"`
	Currency string  `json:"currency"`
}

type TotalPriceDTO struct {
	Amount         float64 `json:"amount"`
	ToPayAmount    float64 `json:"toPayAmount"`
	ToRefundAmount float64 `json:"toRefundAmount"`
}

type AmenityDTO struct {
	Name string `json:"name"`
}

type RoomDTO struct {
	ID         string `json:"id"`
	Name       string `json:"name"`
	RoomTypeID string `json:"roomTypeId"`
}

type RoomStayDTO struct {
	ID                     string             `json:"id"`
	BookingID              string             `json:"bookingId"`
	RoomID                 string             `json:"roomId"`
	RoomTypeID             string             `json:"roomTypeId"`
	CheckInDateTime        *HotelTime         `json:"checkInDateTime"`
	CheckOutDateTime       *HotelTime         `json:"checkOutDateTime"`
	ActualCheckInDateTime  *HotelTime         `json:"actualCheckInDateTime"`
	ActualCheckOutDateTime *HotelTime         `json:"actualCheckOutDateTime"`
	Status                 RoomStayStatus     `json:"status"`
	BookingStatus          BookingStatus      `json:"bookingStatus"`
	GuestCountInfo         *GuestCountInfoDTO `json:"guestCountInfo"`
	Deposit                *AmountDTO         `json:"deposit"`
	TotalPrice             *TotalPriceDTO     `json:"totalPrice"`
	Amenities              []AmenityDTO       `json:"amenities"`
}

type BookingDTO struct {
	ID                string             `json:"id"`
	Number            string             `json:"number"`
	CustomerLanguage  string             `json:"customerLanguage"`
	VisitPurpose      *DictionaryItemDTO `json:"visitPurpose"`
	CustomerComment   string             `json:"customerComment"`
	LastModified      *Timestamp         `json:"lastModified"`
	GroupName         string             `json:"groupName"`
	CurrencyID        string             `json:"currencyId"`
	Customer          *GuestDTO          `json:"customer"`
	CustomerCompany   *CompanyDTO        `json:"customerCompany"`
	RoomStays         []RoomStayDTO      `json:"roomStays"`
	Source            *DictionaryItemDTO `json:"source"`
	SourceChannelName string             `json:"sourceChannelName"`
}

func RoomID(val string) FindBookingsFilter {
	return FindBookingsFilter{
		{
			name:  "roomId",
			value: val,
		},
	}
}

func (t *Timestamp) UnmarshalJSON(b []byte) error {
	s := strings.Trim(string(b), "\"")
	tt, err := time.Parse(time.RFC3339, s)
	if err != nil {
		return err
	}
	t.Time = tt
	return nil
}

func (t *Timestamp) MarshalJSON() ([]byte, error) {
	if t == nil {
		return nil, nil
	}
	return []byte(fmt.Sprintf("%q", t.Format(time.RFC3339))), nil
}

func (t *HotelTime) UnmarshalJSON(b []byte) error {
	s := strings.Trim(string(b), "\"")
	tt, err := time.Parse(hotelTimeLayout, s)
	if err != nil {
		return err
	}
	t.Time = tt
	return nil
}

func (t *HotelTime) MarshalJSON() ([]byte, error) {
	if t == nil {
		return nil, nil
	}
	return []byte(fmt.Sprintf("%q", t.Format(hotelTimeLayout))), nil
}

func (t *Date) UnmarshalJSON(b []byte) error {
	s := strings.Trim(string(b), "\"")
	tt, err := time.Parse(dateLayout, s)
	if err != nil {
		return err
	}
	t.Time = tt
	return nil
}

func (t *Date) MarshalJSON() ([]byte, error) {
	if t == nil {
		return nil, nil
	}
	return []byte(fmt.Sprintf("%q", t.Format(dateLayout))), nil
}

func (t *HotelTime) ZonedTo(location *time.Location) *time.Time {
	if t == nil {
		return nil
	}
	tt := time.Date(t.Year(), t.Month(), t.Day(), t.Hour(), t.Minute(), 0, 0, location)
	return &tt
}
