package main

import (
	"encoding/json"
	"fmt"
	"io/ioutil"
	"net/http"
	"net/url"
)

type Deal struct {
	ID         string `json:"ID"`
	Title      string `json:"TITLE"`
	Stage      string `json:"STAGE_ID"`
	FoodPromo  string `json:"UF_CRM_1622032303674"`
	TaxiPromo  string `json:"UF_CRM_1622032314072"`
	LavkaPromo string `json:"UF_CRM_1622032332118"`
	PlusPromo  string `json:"UF_CRM_1622032352962"`
}

type DealList struct {
	Result []Deal `json:"result"`
	Total  int    `json:"total"`
	Next   *int   `json:"next,omitempty"`
}

type DealUpdateResult struct {
	Result bool `json:"result"`
}

type fieldName string

const checkinTodayStageID = "1"
const checkinTomorrowStageID = "76"
const alreadyStayingStageID = "2"

var stagesToCheck = []string{checkinTomorrowStageID, checkinTodayStageID, alreadyStayingStageID}

func listMatchingDeals(promoField fieldName) ([]Deal, bool, error) {
	var result []Deal
	hasNext := false
	for _, stage := range stagesToCheck {
		r, err := listDealsWithNoPromoCode(promoField, stage)
		if err != nil {
			return nil, false, err
		}
		result = append(result, r.Result...)
		hasNext = hasNext || r.Next != nil
	}
	return result, hasNext, nil
}

func listDealsWithNoPromoCode(promoField fieldName, stage string) (*DealList, error) {
	q := query{
		args:   url.Values{},
		method: "crm.deal.list.json",
	}
	q.fields("ID", "TITLE", "STAGE_ID").
		fields(promoFields...).
		eq("STAGE_ID", stage).
		null(promoField)
	result := DealList{}
	err := httpGet(q, &result)
	if err != nil {
		return nil, err
	}
	return &result, nil
}

func findDealByPromoCode(code string, promoField fieldName) (*Deal, error) {
	q := query{
		args:   url.Values{},
		method: "crm.deal.list.json",
	}
	q.fields("ID", "TITLE").
		eq(promoField, code)
	result := DealList{}
	err := httpGet(q, &result)
	if err != nil {
		return nil, err
	}
	if result.Total == 0 {
		return nil, nil
	} else {
		return &result.Result[0], nil
	}
}

func setPromoCodeToDeal(id string, code string, promoField fieldName) error {
	q := query{
		args:   url.Values{},
		method: "crm.deal.update.json",
	}
	q.args.Add("ID", id)
	q.args.Add(fmt.Sprintf("FIELDS[%s]", promoField), code)
	result := &DealUpdateResult{}
	err := httpPost(q, &result)
	if err != nil {
		return err
	}
	if !result.Result {
		return fmt.Errorf("unable to update code, result is false")
	}
	return nil
}

type query struct {
	args   url.Values
	method string
}

func (q *query) fields(fields ...string) *query {
	for _, v := range fields {
		q.args.Add("SELECT[]", v)
	}
	return q
}

func (q *query) null(field fieldName) *query {
	q.args.Add(fmt.Sprintf("FILTER[=%s]", field), "")
	return q
}

func (q *query) eq(field fieldName, value interface{}) *query {
	q.args.Add(fmt.Sprintf("FILTER[=%s]", field), fmt.Sprintf("%v", value))
	return q
}

func (q *query) ne(field fieldName, value interface{}) *query {
	q.args.Add(fmt.Sprintf("FILTER[!=%s]", field), fmt.Sprintf("%v", value))
	return q
}

func (q *query) gt(field fieldName, value interface{}) *query {
	q.args.Add(fmt.Sprintf("FILTER[>%s]", field), fmt.Sprintf("%v", value))
	return q
}

func (q *query) lt(field fieldName, value interface{}) *query {
	q.args.Add(fmt.Sprintf("FILTER[<%s]", field), fmt.Sprintf("%v", value))
	return q
}

type UnexpectedHTTPStatusError struct {
	StatusCode   int
	ResponseBody *http.Response
}

func (e UnexpectedHTTPStatusError) Error() string {
	return fmt.Sprintf("unexpected HTTP status %d, response is %+v", e.StatusCode, e.ResponseBody)
}

func httpGet(q query, result interface{}) error {
	request, err := http.NewRequest("GET",
		fmt.Sprintf("%s/%s/%s", bitrixPrefix, bitrixSecret.Value, q.method),
		nil)
	if err != nil {
		return fmt.Errorf("error while building http request: %w", err)
	}
	request.URL.RawQuery = q.args.Encode()
	request.Header = http.Header{}
	resp, err := http.DefaultClient.Do(request)
	if err != nil {
		return fmt.Errorf("unable to execute http request: %w", err)
	}
	if resp.StatusCode != 200 {
		return UnexpectedHTTPStatusError{
			StatusCode:   resp.StatusCode,
			ResponseBody: resp,
		}
	}
	data, err := ioutil.ReadAll(resp.Body)
	if err != nil {
		return fmt.Errorf("unable to read response body: %w", err)
	}
	err = json.Unmarshal(data, result)
	return err
}

func httpPost(q query, result interface{}) error {
	resp, err := http.PostForm(fmt.Sprintf("%s/%s/%s", bitrixPrefix, bitrixSecret.Value, q.method), q.args)
	if err != nil {
		return fmt.Errorf("error while posting form: %w", err)
	}
	if resp.StatusCode != 200 {
		return UnexpectedHTTPStatusError{
			StatusCode:   resp.StatusCode,
			ResponseBody: resp,
		}
	}
	data, err := ioutil.ReadAll(resp.Body)
	if err != nil {
		return fmt.Errorf("unable to read response body: %w", err)
	}
	err = json.Unmarshal(data, result)
	return err
}
