package main

import (
	"a.yandex-team.ru/travel/hotels/tools/boy_hotels_checker/pkg/cfg"
	"context"
	"fmt"
	"log"
	"os"
	"sync/atomic"
)

func main() {
	if err := cfg.LoadSettings(getSettings()...); err != nil {
		log.Fatal(err)
	}
	run(context.Background())
}

type PromoType int

const (
	Food PromoType = iota
	Taxi
	Lavka
	Plus
	Hotel
)

var promoTypes = []PromoType{Food, Taxi, Lavka, Plus, Hotel}

var promoFields = []string{foodPromoField, taxiPromoField, lavkaPromoField, plusPromoField}

func (t PromoType) String() string {
	return [...]string{"food", "taxi", "lavka", "plus", "hotel"}[t]
}

func (t PromoType) field() fieldName {
	return [...]fieldName{foodPromoField, taxiPromoField, lavkaPromoField, plusPromoField, hotelPromoField}[t]
}

func run(ctx context.Context) {
	ytClient, err := getYtClient()
	if err != nil {
		panic(err)
	}
	var numErrors int32
	log.Printf("Started")
	for _, pt := range promoTypes {
		if err := transferPromos(ctx, ytClient, pt.String()); err != nil {
			log.Println(fmt.Errorf("unable to transfer new promos: %w", err))
		}
		err := runTillEnd(ctx, pt)
		if err != nil {
			log.Println(fmt.Errorf("error while running routine for type %s: %w", pt.String(), err))
			atomic.AddInt32(&numErrors, 1)
		}
	}
	if numErrors > 0 {
		os.Exit(1)
	}
}

func runTillEnd(ctx context.Context, promoType PromoType) error {
	for {
		hasMore, err := runSingle(ctx, promoType)
		if err != nil {
			return err
		}
		if !hasMore {
			log.Printf("Done applying promocodes of type %s", promoType)
			return nil
		}
		log.Printf("Not all promocodes of type %s are applied, will run once more", promoType)
	}
}

func runSingle(ctx context.Context, promoType PromoType) (bool, error) {
	hasMore := false
	deals, more, err := listMatchingDeals(promoType.field())
	if err != nil {
		return false, err
	}
	if more {
		hasMore = true
	}
	ytClient, err := getYtClient()
	if err != nil {
		return false, err
	}
	log.Printf("Got %d deals to update for type %s", len(deals), promoType)
	tx, err := ytClient.BeginTabletTx(ctx, nil)
	if err != nil {
		return false, err
	}
	promos, err := getPromos(ctx, tx, promoType.String(), len(deals))
	if err != nil {
		return false, fmt.Errorf("unable to get new promocodes: %w", err)
	}
	if len(promos) < len(deals) {
		return false, fmt.Errorf("not enough promocodes of type %s", promoType.String())
	}

	for promoIndex, promo := range promos {
		existing, err := findDealByPromoCode(promo.Code, promoType.field())
		if err != nil {
			return false, fmt.Errorf("unable to check promo for duplicates: %w", err)
		}
		if existing != nil {
			log.Printf("Code %s of type %s is already assigned, expecting duplicate", promo.Code, promo.Type)
			err = updatePromo(ctx, tx, promoType.String(), promo.Code, existing.ID, existing.Title)
			if err != nil {
				return false, fmt.Errorf("unable to duplicate-mark promo in storage: %w", err)
			}
			hasMore = true
		} else {
			deal := deals[promoIndex]
			log.Printf("Setting %s promo code to deal with id %s", promoType.String(), deal.ID)
			err = setPromoCodeToDeal(deal.ID, promo.Code, promoType.field())
			if err != nil {
				return false, fmt.Errorf("unable to apply promode to deal with id %s: %w", deal.ID, err)
			}
			err = updatePromo(ctx, tx, promoType.String(), promo.Code, deal.ID, deal.Title)
			if err != nil {
				return false, fmt.Errorf("unable to update promo in storage: %w", err)
			}
		}
	}
	err = tx.Commit()
	if err != nil {
		return false, fmt.Errorf("unable to commit transaction: %w", err)
	}
	return hasMore, nil
}
