package main

import (
	"context"
	"fmt"
	"log"
	"time"

	"a.yandex-team.ru/yt/go/ypath"
	"a.yandex-team.ru/yt/go/yt"
	"a.yandex-team.ru/yt/go/yt/ythttp"
)

const selectQuery = "* FROM [%s] WHERE type='%s' AND deal_id=null AND expires_at > %d ORDER BY expires_at LIMIT %d"

type availablePromoCode struct {
	Type      string  `yson:"type"`
	Code      string  `yson:"code"`
	ExpiresAt *uint64 `yson:"expires_at"`
}

type appliedPromoCode struct {
	Type      string  `yson:"type"`
	Code      string  `yson:"code"`
	DealID    *string `yson:"deal_id"`
	DealTitle *string `yson:"deal_title"`
	AppliedAt *uint64 `yson:"applied_at"`
}

type promoCodeMinimized struct {
	Code      string  `yson:"code"`
	ExpiresAt *uint64 `yson:"expires_at"`
}

func getYtClient() (yt.Client, error) {
	client, err := ythttp.NewClient(&yt.Config{
		Proxy: ytProxy.Value,
		Token: ytToken.Value,
	})
	if err != nil {
		return nil, err
	}

	return client, nil
}

func getPromos(ctx context.Context, tx yt.TabletTx, promoType string, n int) ([]availablePromoCode, error) {
	now := time.Now()
	nextMonth := now.AddDate(0, 1, 0)
	ts := nextMonth.Unix()
	reader, err := tx.SelectRows(ctx, fmt.Sprintf(selectQuery, ypath.Path(ytPath.Value+"/promos"), promoType, ts, n), &yt.SelectRowsOptions{})
	if err != nil {
		return nil, err
	}
	var res []availablePromoCode
	for reader.Next() {
		var row availablePromoCode
		err = reader.Scan(&row)
		if err != nil {
			return nil, err
		}
		res = append(res, row)
	}
	return res, reader.Err()
}

func updatePromo(ctx context.Context, tx yt.TabletTx, promoType string, code string, dealID string, dealTitle string) error {
	upd := true
	now := uint64(time.Now().Unix())
	toInsert := []interface{}{&appliedPromoCode{
		Type:      promoType,
		Code:      code,
		DealID:    &dealID,
		DealTitle: &dealTitle,
		AppliedAt: &now,
	}}
	return tx.InsertRows(ctx, ypath.Path(ytPath.Value+"/promos"), toInsert, &yt.InsertRowsOptions{
		Update: &upd,
	})
}

func transferPromos(ctx context.Context, client yt.Client, promoType string) error {
	sourcePath := ytPath.Value + "/" + promoType
	sourceTable := ypath.Path(sourcePath)
	var rowCount int
	if err := client.GetNode(ctx, sourceTable.Attr("row_count"), &rowCount, nil); err != nil {
		return err
	}
	if rowCount == 0 {
		log.Printf("No new promocodes of type %s", promoType)
		return nil
	} else {
		log.Printf("Found %d promocodes of type %s, will transfer new to store", rowCount, promoType)

		rdr, err := client.ReadTable(ctx, sourceTable, nil)
		if err != nil {
			return err
		}
		var toInsert []interface{}
		for rdr.Next() {
			var pcm promoCodeMinimized
			if err := rdr.Scan(&pcm); err != nil {
				return err
			}
			toInsert = append(toInsert, availablePromoCode{
				Type:      promoType,
				Code:      pcm.Code,
				ExpiresAt: pcm.ExpiresAt,
			})
		}
		if err := rdr.Err(); err != nil {
			return err
		}

		upd := true
		if err := client.InsertRows(ctx, ypath.Path(ytPath.Value+"/promos"), toInsert, &yt.InsertRowsOptions{
			Update: &upd,
		}); err != nil {
			return err
		}
		spec := map[string]interface{}{
			"table_path": sourceTable,
		}
		opID, err := client.StartOperation(ctx, yt.OperationErase, spec, nil)
		if err != nil {
			return err
		} else {
			log.Printf("Started erase operation %s on table %s", opID.String(), sourcePath)
			return nil
		}
	}
}
