package main

import (
	"context"
	"fmt"
	"sync"

	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/library/go/maxprocs"
	"a.yandex-team.ru/travel/library/go/metrics"
	metricsserver "a.yandex-team.ru/travel/library/go/metrics/server"

	"a.yandex-team.ru/travel/buses/backend/internal/api/app"
	"a.yandex-team.ru/travel/buses/backend/internal/api/config"
	"a.yandex-team.ru/travel/buses/backend/internal/api/grpcserver"
	"a.yandex-team.ru/travel/buses/backend/internal/api/httpserver"
	"a.yandex-team.ru/travel/buses/backend/internal/common/logging"
)

func main() {
	maxprocs.AdjustAuto()

	cfg, err := config.Load()
	if err != nil {
		fmt.Println("can not load configuration:", err)
		return
	}

	logger, err := logging.New(&cfg.Logging)
	if err != nil {
		fmt.Println("failed to create logger, err:", err)
		return
	}
	defer func() { _ = logger.L.Sync() }()

	metricsRegistry := metrics.NewRegistryWithDeployTagsAndExplicitHost()
	metrics.RunPerfMetricsUpdater(metricsRegistry, cfg.Metrics.PerfMetricsRefreshInterval)

	app, err := app.NewApp(&cfg.App, logger, metricsRegistry.WithPrefix("app"))
	if err != nil {
		logger.Error("failed to create App", log.Error(err))
		return
	}
	if err = app.Run(); err != nil {
		logger.Error("failed to run App", log.Error(err))
		return
	}
	defer app.Close()

	wg := sync.WaitGroup{}

	grpcServer := grpcserver.New(app, &cfg.GRPC, logger, metricsRegistry.WithPrefix("grpc"))

	wg.Add(1)
	go func() {
		defer wg.Done()
		if err = grpcServer.Run(context.Background()); err != nil {
			logger.Fatal("error while starting GRPC server", log.Error(err))
		}
	}()

	httpServer, err := httpserver.New(app, &cfg.HTTP, logger, metricsRegistry.WithPrefix("http"))
	if err != nil {
		logger.Error("failed to create HTTP server", log.Error(err))
		return
	}

	wg.Add(1)
	go func() {
		defer wg.Done()
		defer func() { _ = httpServer.Close() }()
		if err = httpServer.ListenAndServe(); err != nil {
			logger.Fatal("error while starting HTTP server", log.Error(err))
		}
	}()

	wg.Add(1)
	go func() {
		defer wg.Done()
		if err = metricsserver.RunMetricsHTTPServer(context.Background(), cfg.Metrics, logger, metricsRegistry); err != nil {
			logger.Fatal("error while starting metrics server", log.Error(err))
		}
	}()

	wg.Wait()
}
