package main

import (
	"context"
	"fmt"
	"sync"

	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/library/go/maxprocs"
	"a.yandex-team.ru/travel/library/go/configuration"
	"a.yandex-team.ru/travel/library/go/grpcgateway"
	"a.yandex-team.ru/travel/library/go/metrics"
	metricServer "a.yandex-team.ru/travel/library/go/metrics/server"
	"a.yandex-team.ru/travel/library/go/tracing"

	"a.yandex-team.ru/travel/buses/backend/internal/common/grpc"
	"a.yandex-team.ru/travel/buses/backend/internal/common/logging"
	"a.yandex-team.ru/travel/buses/backend/internal/worker/app"
	wpb "a.yandex-team.ru/travel/buses/backend/proto/worker"
)

type Config struct {
	App     *app.Config
	GRPC    *grpc.ServerConfig
	Gateway *grpcgateway.Config
	Logging *logging.Config
	Metrics *metricServer.MetricsConfig
}

var cfg = Config{
	App:     &app.DefaultConfig,
	GRPC:    &grpc.DefaultServerConfig,
	Gateway: &grpcgateway.DefaultConfig,
	Logging: &logging.DefaultConfig,
	Metrics: &metricServer.DefaultMetricsConfig,
}

const (
	swaggerJSONName = "worker_service.swagger.json"
	swaggerPrefix   = "/worker"
)

func main() {
	maxprocs.AdjustAuto()

	ctx, cancel := context.WithCancel(context.Background())
	defer cancel()

	loader := configuration.NewDefaultConfitaLoader()
	err := loader.Load(ctx, &cfg)
	if err != nil {
		fmt.Println("failed to load config:", err)
		return
	}
	logger, err := logging.New(cfg.Logging)
	if err != nil {
		logger.Error("failed to create logger", log.Error(err))
		return
	}

	tracerCloser := tracing.InitializeDefaultTracer("buses_backend.worker")
	defer func() {
		err = tracerCloser.Close()
		if err != nil {
			logger.Error("tracer close error:", log.Error(err))
		}
		err = logger.L.Sync()
		if err != nil {
			fmt.Println("failed to close logger:", err)
			return
		}
	}()

	metricsRegistry := metrics.NewRegistryWithDeployTagsAndExplicitHost()
	metrics.RunPerfMetricsUpdater(metricsRegistry, cfg.Metrics.PerfMetricsRefreshInterval)

	app, err := app.NewApp(cfg.App, metricsRegistry.WithPrefix("app"), logger)
	if err != nil {
		logger.Error("failed to create App", log.Error(err))
		return
	}
	if err = app.Run(); err != nil {
		logger.Fatal("failed to run App", log.Error(err))
		return
	}
	defer app.Close()

	grpcServer, err := grpc.NewServer(cfg.GRPC, metricsRegistry.WithPrefix("grpc"), logger)
	if err != nil {
		logger.Fatal("failed to create grpc server", log.Error(err))
		return
	}
	wpb.RegisterWorkerServiceServer(grpcServer.GetCoreServer(), app)

	wg := sync.WaitGroup{}

	wg.Add(1)
	go func() {
		defer wg.Done()
		if err = grpcServer.Serve(); err != nil {
			logger.Fatal("GRPC server closed", log.Error(err))
		}
	}()

	wg.Add(1)
	go func() {
		defer wg.Done()
		err = metricServer.RunMetricsHTTPServer(context.Background(), *cfg.Metrics, logger, metricsRegistry)
		if err != nil {
			logger.Fatal("Metrics server closed", log.Error(err))
		}
	}()

	if cfg.Gateway.Enabled {
		wg.Add(1)
		go func() {
			defer wg.Done()

			grpcGateway := grpcgateway.NewGateway(cfg.Gateway, grpcgateway.NewService(swaggerJSONName, swaggerPrefix, cfg.GRPC.Address, wpb.RegisterWorkerServiceHandlerFromEndpoint, nil))
			logger.Infof("GRPC-gateway server starting at: %s", cfg.Gateway.Address)
			err = grpcGateway.Run(ctx)
			if err != nil {
				logger.Fatal("GRPC-gateway server closed", log.Error(err))
			}
		}()
	} else {
		logger.Warn("GRPC-gateway server disabled")
	}

	logger.Info("started")
	wg.Wait()
}
