package app

import (
	"a.yandex-team.ru/travel/buses/backend/internal/common/redir"
	"context"
	"fmt"
	"path/filepath"
	"sync"
	"time"

	"a.yandex-team.ru/library/go/core/log/zap"
	coreMetrics "a.yandex-team.ru/library/go/core/metrics"
	"a.yandex-team.ru/travel/library/go/logbroker"
	"a.yandex-team.ru/travel/library/go/metrics"
	"a.yandex-team.ru/travel/library/go/resourcestorage"
	"a.yandex-team.ru/travel/library/go/vault"
	"a.yandex-team.ru/yt/go/yt"
	"go.uber.org/atomic"
	"google.golang.org/grpc"

	apiBlacklist "a.yandex-team.ru/travel/buses/backend/internal/api/blacklist"
	"a.yandex-team.ru/travel/buses/backend/internal/api/cache"
	"a.yandex-team.ru/travel/buses/backend/internal/api/filters"
	"a.yandex-team.ru/travel/buses/backend/internal/api/offerstorage"
	"a.yandex-team.ru/travel/buses/backend/internal/api/warmer"
	"a.yandex-team.ru/travel/buses/backend/internal/common/connector"
	dictRasp "a.yandex-team.ru/travel/buses/backend/internal/common/dict/rasp"
	commonGRPC "a.yandex-team.ru/travel/buses/backend/internal/common/grpc"
	ilogbroker "a.yandex-team.ru/travel/buses/backend/internal/common/logbroker"
	"a.yandex-team.ru/travel/buses/backend/internal/common/logging"
	"a.yandex-team.ru/travel/buses/backend/internal/common/popular"
	ipb "a.yandex-team.ru/travel/buses/backend/internal/common/proto"
	ytclient "a.yandex-team.ru/travel/buses/backend/internal/common/yt"
	pb "a.yandex-team.ru/travel/buses/backend/proto"
	wpb "a.yandex-team.ru/travel/buses/backend/proto/worker"
)

type Config struct {
	FrontSearchTimeout    time.Duration `config:"app-frontsearchtimeout,required"`
	SearchFailedTimeout   time.Duration `config:"app-searchfailedtimeout,required"`
	SearchCacheTTL        time.Duration `config:"app-searchcachettl,required"`
	SegmentsCacheTTL      time.Duration `config:"app-segmentscachettl,required"`
	SegmentsFailedTimeout time.Duration `config:"app-segmentsfailedtimeout,required"`
	SegmentsFromLBMaxAge  time.Duration `config:"app-segmentsfromlbmaxage,required"`
	// Коэффициент TTL для мастера. TODO: Убрать после перехода на CSLB воркера
	MasterSegmentsTTLRatio float64       `config:"app-mastersegmentsttlratio,required"`
	DoNotDisturbPeriod     time.Duration `config:"app-donotdisturbperiod"`
	BannedRidesTTL         time.Duration `config:"app-bannedridesttl,required"`
	Suppliers              []uint32      `config:"app-suppliers,required"`
	FrontBaseURL           string        `config:"app-frontbaseurl,required"`
	FrontFaviconURL        string        `config:"app-frontfaviconurl,required"`
	DefaultTimezone        string        `config:"app-defaulttimezone,required"`
	DumpPeriod             time.Duration `config:"app-dumpperiod,required"`
	DumpKeepLastVersions   int           `config:"app-dumpkeeplastversions,required"`
	S3StorageSecret        string        `config:"app-s3storagesecret,required"`
	S3StorageAccessKeyKey  string        `config:"app-s3storageaccesskeykey,required"`
	S3StorageSecretKey     string        `config:"app-s3storagesecretkey,required"`
	LogbrokerConsumer      ilogbroker.ConsumerConfig
	Cache                  cache.Config
	Storage                resourcestorage.S3StorageConfig
	Connector              connector.Config
	Billing                BillingConfig
	Popular                popular.Config
	DictRasp               dictRasp.Config
	Blacklist              apiBlacklist.Config
	Warmer                 warmer.Config
	Worker                 commonGRPC.ClientConfig
	YtLockClient           ytclient.ClientConfig
	OfferStorage           offerstorage.Config
	Redir                  redir.Config
}

var DefaultConfig = Config{
	FrontSearchTimeout:     15 * time.Second,
	SearchFailedTimeout:    15 * time.Minute,
	SearchCacheTTL:         10 * time.Hour,
	SegmentsCacheTTL:       20 * time.Hour,
	SegmentsFailedTimeout:  1 * time.Hour,
	SegmentsFromLBMaxAge:   20 * time.Hour,
	MasterSegmentsTTLRatio: 0.95,
	BannedRidesTTL:         10 * time.Hour,
	Suppliers:              []uint32{},
	FrontBaseURL:           "https://yandex.%s/bus",
	FrontFaviconURL:        "https://yastatic.net/q/bus/v0/static/desktop/favicon.ico",
	DefaultTimezone:        "Europe/Moscow",
	DumpPeriod:             15 * time.Minute,
	DumpKeepLastVersions:   100,
	LogbrokerConsumer:      ilogbroker.DefaultDeployConsumerConfig,
	Cache:                  cache.DefaultConfig,
	Storage:                resourcestorage.DefaultS3StorageConfig,
	Connector:              connector.DefaultConfig,
	Billing:                DefaultBillingConfig,
	Popular:                popular.DefaultConfig,
	DictRasp:               dictRasp.DefaultConfig,
	Blacklist:              apiBlacklist.DefaultConfig,
	Warmer:                 warmer.DefaultConfig,
	Worker:                 commonGRPC.DefaultClientConfig,
	YtLockClient:           ytclient.DefaultConfig,
	OfferStorage:           offerstorage.DefaultConfig,
	Redir:                  redir.DefaultConfig,
}

type BillingConfig struct {
	DictPath string `config:"billing-dictpath,required"`
}

var DefaultBillingConfig = BillingConfig{
	DictPath: "/app/billing-dict.yaml",
}

const (
	searchRecordStorageResource = "search_cache"
	popularDirectionsResource   = "popular_directions"
	moduleName                  = "app"
)

type App struct {
	logger    *zap.Logger
	cfg       *Config
	ctx       context.Context
	ctxCancel context.CancelFunc

	onAir        *atomic.Bool
	isMaster     bool
	ytLockClient yt.Client

	metricsRegistry       coreMetrics.Registry
	appMetrics            *metrics.AppMetrics
	searchMetricHistogram coreMetrics.Histogram

	searchConsumer   *logbroker.Consumer
	segmentsConsumer *logbroker.Consumer

	searchCacheWarmer *warmer.Warmer

	raspRepo                *dictRasp.DictRepo
	searchCache             *cache.SearchRecordStorage
	searchCacheDumper       *resourcestorage.Dumper
	popularDirections       *popular.Directions
	popularDirectionsDumper *resourcestorage.Dumper
	segmentsCache           *cache.SegmentsStorage

	billingDict *BillingData

	bannedRidesRule *filters.BannedRidesRule
	blacklist       *apiBlacklist.Blacklist
	ridesFilter     *filters.RidesFilter

	workerServiceConnection *grpc.ClientConn
	workerServiceClient     wpb.WorkerServiceClient

	offerStorage *offerstorage.Storage
	redir        *redir.Service

	segmentsMutex sync.RWMutex
}

func loadBillingDict(config *BillingConfig) (*BillingData, error) {
	billingDictPath, err := filepath.Abs(config.DictPath)
	if err != nil {
		return nil, err
	}
	billingDict, err := LoadBillingData(billingDictPath)
	if err != nil {
		return nil, err
	}
	return billingDict, nil
}

func NewApp(cfg *Config, logger *zap.Logger, metricsRegistry coreMetrics.Registry) (*App, error) {
	const errorMessageFormat = "NewApp fails: %w"
	moduleLogger := logging.WithModuleContext(logger, moduleName)

	searchMetricHistogram := metricsRegistry.Histogram("search.rides_hist",
		coreMetrics.NewBuckets(0, 2, 5, 10, 20, 40))
	appMetrics := metrics.NewAppMetrics(metricsRegistry)

	raspRepo := dictRasp.NewRepo(&cfg.DictRasp, moduleLogger)
	if err := raspRepo.Load(); err != nil {
		return nil, fmt.Errorf(errorMessageFormat, err)
	}

	searchCache := cache.NewSearchRecordStorage(cfg.SearchCacheTTL, appMetrics, &cfg.Cache, moduleLogger)
	popularDirections := popular.NewDirections(cfg.Popular, moduleLogger)
	segmentsCache := cache.NewSegmentsStorage(appMetrics)

	defaultTimeLocation, err := time.LoadLocation(cfg.DefaultTimezone)
	if err != nil {
		return nil, fmt.Errorf(errorMessageFormat, err)
	}
	bannedRidesRule := filters.NewBannedRidesRule(cfg.BannedRidesTTL)
	blacklist := apiBlacklist.NewBlacklist(&cfg.Blacklist, logger)
	ridesFilter := filters.NewRidesFilter(bannedRidesRule, blacklist, &filters.FreeSeatsRule{},
		&filters.RideStatusRule{}, filters.NewDepartureInFutureRule(raspRepo, defaultTimeLocation))

	secretResolver := vault.NewYavSecretsResolver()

	var searchCacheDumper *resourcestorage.Dumper
	var popularDirectionsDumper *resourcestorage.Dumper
	var searchCacheWarmer *warmer.Warmer

	var s3StorageAccessKey, s3StorageSecret string
	if cfg.Storage != resourcestorage.MockedS3StorageConfig { // check if S3 storage disabled
		s3StorageAccessKey, err = secretResolver.GetSecretValue(cfg.S3StorageSecret, cfg.S3StorageAccessKeyKey)
		if err != nil {
			return nil, fmt.Errorf(errorMessageFormat, err)
		}
		s3StorageSecret, err = secretResolver.GetSecretValue(cfg.S3StorageSecret, cfg.S3StorageSecretKey)
		if err != nil {
			return nil, fmt.Errorf(errorMessageFormat, err)
		}
	}

	storageReader := resourcestorage.NewS3StorageReader(cfg.Storage, s3StorageAccessKey, s3StorageSecret)

	searchConsumerTS, err := storageReader.GetTimestamp(searchRecordStorageResource)
	if err != nil {
		searchConsumerTS = time.Now().Add(-cfg.SearchCacheTTL)
		moduleLogger.Errorf("Can not get snapshot timestamp: %s", err.Error())
	} else {
		searchCacheLoader := resourcestorage.NewLoader(&ipb.TSearchCacheRecord{},
			searchRecordStorageResource, storageReader, moduleLogger)
		n, err := searchCacheLoader.Load(searchCache)
		if err != nil {
			moduleLogger.Errorf("Can not load snapshot for %s: %s", searchRecordStorageResource, err.Error())
		} else {
			moduleLogger.Infof("Loaded %d records for %s", n, searchRecordStorageResource)
		}
	}
	popularDirectionsLoader := resourcestorage.NewLoader(&ipb.TPopularDirection{},
		popularDirectionsResource, storageReader, moduleLogger)
	n, err := popularDirectionsLoader.Load(popularDirections)
	if err != nil {
		moduleLogger.Errorf("Can not load snapshot for %s: %s", popularDirectionsResource, err.Error())
	} else {
		moduleLogger.Infof("Loaded %d records for %s", n, popularDirectionsResource)
	}

	ytLockClient, err := ytclient.NewYtLockClient(cfg.YtLockClient, secretResolver)
	if err != nil {
		return nil, fmt.Errorf("can not create yt lock client: %w", err)
	}

	searchConsumer, err := ilogbroker.NewConsumer(
		cfg.LogbrokerConsumer, ilogbroker.SearchResultTopic, cfg.LogbrokerConsumer.SearchYtLockPrefix, searchConsumerTS,
		secretResolver, ytLockClient, logger)
	if err != nil {
		return nil, fmt.Errorf("can not create search consumer: %w", err)
	}
	segmentsConsumer, err := ilogbroker.NewConsumer(
		cfg.LogbrokerConsumer, ilogbroker.SegmentsResultTopic, cfg.LogbrokerConsumer.SegmentsYtLockPrefix,
		time.Now().Add(-cfg.SegmentsFromLBMaxAge), secretResolver, ytLockClient, logger)
	if err != nil {
		return nil, fmt.Errorf("can not create segments consumer: %w", err)
	}
	searchConsumer.RegisterMetrics(appMetrics, "search_consumer")
	segmentsConsumer.RegisterMetrics(appMetrics, "segments_consumer")

	billingDict, err := loadBillingDict(&cfg.Billing)
	if err != nil {
		return nil, fmt.Errorf("can not load billing dict: %w", err)
	}

	ctx, ctxCancel := context.WithCancel(context.Background())
	offerStorage, err := offerstorage.NewStorage(ctx, &cfg.OfferStorage, logger)
	if err != nil {
		ctxCancel()
		return nil, fmt.Errorf(errorMessageFormat, err)
	}

	redirService := redir.NewService(&cfg.Redir, logger)

	workerServiceConnection, err := commonGRPC.NewServiceConnection(&cfg.Worker, logger, ctx)
	if err != nil {
		ctxCancel()
		return nil, fmt.Errorf("can not connect to worker at %s: %w", cfg.Worker.Address, err)
	}
	go func() {
		<-ctx.Done()
		_ = workerServiceConnection.Close()
	}()

	workerServiceClient := wpb.NewWorkerServiceClient(workerServiceConnection)

	isMaster := ytclient.AcquireMaster(cfg.YtLockClient, ytLockClient, ctx)
	if isMaster {
		moduleLogger.Info("Started as master")
		searchCacheDumper = resourcestorage.NewDumper(
			searchCache, searchRecordStorageResource,
			resourcestorage.NewS3StorageWriter(cfg.Storage, s3StorageAccessKey, s3StorageSecret),
			cfg.DumpKeepLastVersions, moduleLogger)
		popularDirectionsDumper = resourcestorage.NewDumper(
			popularDirections, popularDirectionsResource,
			resourcestorage.NewS3StorageWriter(cfg.Storage, s3StorageAccessKey, s3StorageSecret),
			cfg.DumpKeepLastVersions, moduleLogger)
		searchCacheWarmer = warmer.NewWarmer(cfg.Warmer, searchCache, segmentsCache, workerServiceClient, cfg.Storage,
			s3StorageAccessKey, s3StorageSecret, raspRepo, appMetrics, cfg.Suppliers, moduleLogger)
		cfg.SegmentsCacheTTL = time.Duration(cfg.SegmentsCacheTTL.Seconds()*cfg.MasterSegmentsTTLRatio) * time.Second
		cfg.SegmentsFailedTimeout = time.Duration(cfg.SegmentsFailedTimeout.Seconds()*cfg.MasterSegmentsTTLRatio) * time.Second
	}

	return &App{
		logger:                  moduleLogger,
		ctx:                     ctx,
		ctxCancel:               ctxCancel,
		isMaster:                isMaster,
		onAir:                   atomic.NewBool(false),
		ytLockClient:            ytLockClient,
		metricsRegistry:         metricsRegistry,
		appMetrics:              appMetrics,
		searchMetricHistogram:   searchMetricHistogram,
		cfg:                     cfg,
		searchConsumer:          searchConsumer,
		segmentsConsumer:        segmentsConsumer,
		raspRepo:                raspRepo,
		searchCache:             searchCache,
		searchCacheDumper:       searchCacheDumper,
		searchCacheWarmer:       searchCacheWarmer,
		popularDirections:       popularDirections,
		popularDirectionsDumper: popularDirectionsDumper,
		segmentsCache:           segmentsCache,
		billingDict:             billingDict,
		bannedRidesRule:         bannedRidesRule,
		ridesFilter:             ridesFilter,
		blacklist:               blacklist,
		workerServiceConnection: workerServiceConnection,
		workerServiceClient:     workerServiceClient,
		offerStorage:            offerStorage,
		redir:                   redirService,
	}, nil
}

func (a *App) Run() error {
	a.searchCache.Run(a.ctx)
	if a.searchCacheDumper != nil {
		a.searchCacheDumper.RunPeriodic(a.cfg.DumpPeriod, a.ctx)
	}
	if a.popularDirectionsDumper != nil {
		a.popularDirectionsDumper.RunPeriodic(a.cfg.DumpPeriod, a.ctx)
	}
	if a.searchCacheWarmer != nil {
		a.searchCacheWarmer.Run(a.ctx)
	}
	a.popularDirections.Run(a.ctx)

	if err := a.searchConsumer.Run(a.ctx); err != nil {
		return err
	}
	go a.searchConsumerLoop()

	if err := a.segmentsConsumer.Run(a.ctx); err != nil {
		return err
	}
	go a.segmentsConsumerLoop()

	a.bannedRidesRule.Run(a.ctx)
	a.blacklist.RunFetcher(a.ctx)

	go func() {
		time.Sleep(a.cfg.DoNotDisturbPeriod)
		a.onAir.Store(true)
	}()

	return nil
}

func (a *App) Close() {
	a.ctxCancel()
}

func (a *App) GetMetricsRegistry() coreMetrics.Registry {
	return a.metricsRegistry
}

func (a *App) ReloadResources() *StatusWithMessage {
	if a.raspRepo != nil {
		if err := a.raspRepo.Load(); err != nil {
			return NewStatusWithMessage(pb.EStatus_STATUS_EXTERNAL_DICT_NOT_FOUND, err.Error())
		}
	}
	return NewStatusWithMessage(pb.EStatus_STATUS_OK, "")
}
