package app

import (
	"context"
	"fmt"
	"time"

	tpb "a.yandex-team.ru/travel/proto"
	"github.com/golang/protobuf/proto"

	"a.yandex-team.ru/travel/buses/backend/internal/common/utils"
	pb "a.yandex-team.ru/travel/buses/backend/proto"
)

const CalendarMaxDays = 31 * 6

func (a *App) Calendar(
	from *pb.TPointKey, to *pb.TPointKey, minDate *tpb.TDate, maxDate *tpb.TDate,
	source pb.ERequestSource, ctx context.Context,
) ([]CalendarItem, error) {
	var (
		minTime = utils.ConvertProtoDateToTime(minDate)
		maxTime = utils.ConvertProtoDateToTime(maxDate)

		result []CalendarItem

		emptyCount, notEmptyCount, notCachedCount int64
	)
	if maxTime.Sub(minTime)/time.Hour/24 > CalendarMaxDays {
		return result, fmt.Errorf("more than %v days requested", CalendarMaxDays)
	}

	for date := minTime; !date.After(maxTime); date = date.AddDate(0, 0, 1) {
		var (
			rides, ready = a.findRides(
				from, to, utils.ConvertTimeToProtoDate(date), true,
				source, ctx,
			)
			found = len(rides) > 0
		)

		if ready || found {
			item := CalendarItem{Date: date}
			if found {
				item.add(rides)
				notEmptyCount++
			} else {
				emptyCount++
			}
			result = append(result, item)
		} else {
			notCachedCount++
		}
	}

	a.addCalendarMetrics(source, pb.EStatus_STATUS_OK, notEmptyCount)
	a.addCalendarMetrics(source, pb.EStatus_STATUS_NOT_FOUND, emptyCount)
	a.addCalendarMetrics(source, pb.EStatus_STATUS_NOT_READY, notCachedCount)

	return result, nil
}

func (a *App) addCalendarMetrics(source pb.ERequestSource, status pb.EStatus, count int64) {
	tagsFound := map[string]string{
		"status": status.String(),
		"source": source.String(),
	}
	a.appMetrics.GetOrCreateCounter("calendar", tagsFound, "day_status").Add(count)
}

type CalendarItem struct {
	Date                   time.Time
	RideCount              uint32
	MinPrices              []*tpb.TPrice
	MinPricesBySupplierIDs map[uint32][]*tpb.TPrice
}

func updatePrices(prices []*tpb.TPrice, price *tpb.TPrice) []*tpb.TPrice {
	for _, p := range prices {
		if p.Currency != price.Currency {
			continue
		}
		if p.Amount > price.Amount {
			p.Amount = price.Amount
		}
		return prices
	}
	return append(prices, proto.Clone(price).(*tpb.TPrice))
}

func (i *CalendarItem) add(rides []*pb.TRide) {
	rideCount := len(rides)
	if rideCount == 0 {
		return
	}

	if i.RideCount == 0 {
		i.MinPricesBySupplierIDs = map[uint32][]*tpb.TPrice{}
	}
	i.RideCount += uint32(rideCount)

	for _, ride := range rides {
		i.MinPrices = updatePrices(i.MinPrices, ride.Price)

		supplierID := ride.SupplierId
		i.MinPricesBySupplierIDs[supplierID] = updatePrices(i.MinPricesBySupplierIDs[supplierID], ride.Price)
	}
}
