package app

/*
Эта часть пакета для поддержки легаси фронта
*/

import (
	"fmt"
	"strconv"
	"strings"

	tpb "a.yandex-team.ru/travel/proto"
	"a.yandex-team.ru/travel/proto/dicts/rasp"

	"a.yandex-team.ru/travel/buses/backend/internal/common/utils"
	pb "a.yandex-team.ru/travel/buses/backend/proto"
)

type Point struct {
	Name        string  `json:"name"`
	Latitude    float64 `json:"latitude,omitempty"`
	Longitude   float64 `json:"longitude,omitempty"`
	Timezone    string  `json:"timezone,omitempty"`
	Code        string  `json:"code"`
	Description string  `json:"description"`
	BestGeoID   int32   `json:"bestGeoId,omitempty"`
	Parent      int32   `json:"parent,omitempty"`
}

type PointsData struct {
	Data []Point `json:"data"`
}

type geoEntity interface {
	GetTitle() *tpb.TTranslationCase
	GetTitleDefault() string
}

func makeGeoEntityTitle(entity geoEntity) string {
	nominative := entity.GetTitle().GetRu().GetNominative()
	if nominative == "" {
		return nominative
	}
	return entity.GetTitleDefault()
}

func makeStationTitle(station *rasp.TStation) string {
	nominative := station.GetTitle().GetRu()
	if nominative == "" {
		return nominative
	}
	return station.GetTitleDefault()
}

func makeRegionTitle(region *rasp.TRegion) string {
	nominative := region.GetTitleNominative().GetRu()
	if nominative == "" {
		return nominative
	}
	return region.GetTitleDefault()
}

func (a *App) makeSettlementDescription(settlement *rasp.TSettlement) string {
	var description []string
	prefix := ""
	isDisputedTerritory := false
	if settlement.Majority == rasp.TSettlement_MAJORITY_CAPITAL ||
		settlement.Majority == rasp.TSettlement_MAJORITY_REGION_CAPITAL ||
		settlement.Majority == rasp.TSettlement_MAJORITY_POPULATION_MILLION ||
		settlement.Majority == rasp.TSettlement_MAJORITY_COMMON_CITY {
		prefix = "г. "
	}
	description = append(description, fmt.Sprintf("%s%s", prefix, makeGeoEntityTitle(settlement)))
	if district, found := a.raspRepo.GetDistrict(settlement.DistrictId); found {
		description = append(description, district.TitleDefault)
		isDisputedTerritory = isDisputedTerritory || district.IsDisputedTerritory
	}
	if region, found := a.raspRepo.GetRegion(settlement.RegionId); found {
		description = append(description, makeRegionTitle(region))
		isDisputedTerritory = isDisputedTerritory || region.IsDisputedTerritory
	}
	if !isDisputedTerritory {
		if country, found := a.raspRepo.GetCountry(settlement.CountryId); found {
			description = append(description, makeGeoEntityTitle(country))
		}
	}
	return strings.Join(description, ", ")
}

func (a *App) makePointBySettlement(settlement *rasp.TSettlement) *Point {
	code, _ := utils.DumpPointKey(
		&pb.TPointKey{
			Type: pb.EPointKeyType_POINT_KEY_TYPE_SETTLEMENT,
			Id:   uint32(settlement.Id),
		},
	)
	tz, _ := a.raspRepo.GetTimeZone(settlement.TimeZoneId)
	return &Point{
		Name:        makeGeoEntityTitle(settlement),
		Latitude:    settlement.Latitude,
		Longitude:   settlement.Longitude,
		Timezone:    tz.GetCode(),
		Code:        code,
		Description: a.makeSettlementDescription(settlement),
		BestGeoID:   settlement.GeoId,
	}
}

func (a *App) makeStationDescription(station *rasp.TStation) string {
	prefix := ""
	if station.Type == rasp.TStation_TYPE_BUS_STATION {
		prefix = "авт.вкз. "
	} else if station.Type == rasp.TStation_TYPE_BUS_STOP {
		prefix = "авт.ост. "
	}
	if settlement, found := a.raspRepo.GetSettlement(station.SettlementId); found {
		return fmt.Sprintf("%s%s, %s", prefix, makeStationTitle(station), a.makeSettlementDescription(settlement))
	}
	return fmt.Sprintf("%s%s", prefix, makeStationTitle(station))
}

func (a *App) makePointByStation(station *rasp.TStation) *Point {
	code, _ := utils.DumpPointKey(
		&pb.TPointKey{
			Type: pb.EPointKeyType_POINT_KEY_TYPE_STATION,
			Id:   uint32(station.Id),
		},
	)
	tz, _ := a.raspRepo.GetTimeZone(station.TimeZoneId)
	return &Point{
		Name:        station.TitleDefault,
		Latitude:    station.Latitude,
		Longitude:   station.Longitude,
		Timezone:    tz.GetCode(),
		Code:        code,
		Description: a.makeStationDescription(station),
		Parent:      -1,
	}
}

func (a *App) GetPointByPointkey(pointkey string) (*Point, *StatusWithMessage) {
	pk, err := utils.LoadPointKey(pointkey)
	if err != nil {
		return nil, NewStatusWithMessage(pb.EStatus_STATUS_NOT_FOUND, err.Error())
	}

	if pk.Type == pb.EPointKeyType_POINT_KEY_TYPE_SETTLEMENT {
		settlement, found := a.raspRepo.GetSettlement(int32(pk.Id))
		if !found {
			return nil, NewStatusWithMessage(pb.EStatus_STATUS_NOT_FOUND, "no such settlement")
		}
		return a.makePointBySettlement(settlement), NewStatusWithMessage(pb.EStatus_STATUS_OK, "")
	} else if pk.Type == pb.EPointKeyType_POINT_KEY_TYPE_STATION {
		station, found := a.raspRepo.GetStation(int32(pk.Id))
		if !found {
			return nil, NewStatusWithMessage(pb.EStatus_STATUS_NOT_FOUND, "no such station")
		}
		return a.makePointByStation(station), NewStatusWithMessage(pb.EStatus_STATUS_OK, "")
	}
	return nil, NewStatusWithMessage(pb.EStatus_STATUS_NOT_FOUND, "unsupported pointkey type")
}

func (a *App) getPointByGeoID(geoID string) (*Point, *StatusWithMessage) {
	id, err := strconv.Atoi(geoID)
	if err != nil {
		return nil, NewStatusWithMessage(pb.EStatus_STATUS_BAD_REQUEST, fmt.Sprintf("bad id: %s", err.Error()))
	}
	settlement, found := a.raspRepo.GetSettlementByGeoID(int32(id))
	if !found {
		return nil, NewStatusWithMessage(pb.EStatus_STATUS_NOT_FOUND, "no such settlement")
	}
	return a.makePointBySettlement(settlement), NewStatusWithMessage(pb.EStatus_STATUS_OK, "")
}

func (a *App) GetPointsWhere(where string) (*PointsData, *StatusWithMessage) {
	whereParts := strings.Split(where, ":")
	if len(whereParts) != 2 {
		return nil, NewStatusWithMessage(pb.EStatus_STATUS_BAD_REQUEST, "where=field:values")
	}
	searchType := whereParts[0]
	points := make([]Point, 0)
	for _, key := range strings.Split(whereParts[1], ",") {
		if searchType == "code" {
			point, status := a.GetPointByPointkey(key)
			if status.Ok() {
				points = append(points, *point)
			}
		} else if searchType == "best_geo_id" {
			point, status := a.getPointByGeoID(key)
			if status.Ok() {
				points = append(points, *point)
			}
		} else {
			return nil, NewStatusWithMessage(pb.EStatus_STATUS_BAD_REQUEST,
				fmt.Sprintf("unknown search type '%s'", searchType))
		}
	}

	return &PointsData{Data: points}, NewStatusWithMessage(pb.EStatus_STATUS_OK, "")
}
