package app

import (
	"context"
	"time"

	tpb "a.yandex-team.ru/travel/proto"
	rpb "a.yandex-team.ru/travel/proto/dicts/rasp"

	ipb "a.yandex-team.ru/travel/buses/backend/internal/common/proto"
	"a.yandex-team.ru/travel/buses/backend/internal/common/utils"
	pb "a.yandex-team.ru/travel/buses/backend/proto"
)

func (a *App) GetPopularDirectionsFrom(
	from *pb.TPointKey, limit int, ctx context.Context,
) *ipb.TPopularDirectionsResponse {
	const (
		searchWidth  = 2 * 24 * time.Hour
		errorMessage = "App.GetPopularDirections"
	)
	var directions []*ipb.TPopularDirection
	if from == nil {
		directions = a.popularDirections.GetDirections()
	} else {
		directions = a.popularDirections.GetDirectionsFrom(from)
	}
	response := &ipb.TPopularDirectionsResponse{
		Directions: make([]*ipb.TPopularDirectionResponse, limit),
	}
	counter := 0
	minDate := time.Now()
	maxDate := minDate.Add(searchWidth)
	for _, direction := range directions {
		if counter >= limit {
			break
		}
		if direction.From == nil || direction.To == nil ||
			direction.From.Type != pb.EPointKeyType_POINT_KEY_TYPE_SETTLEMENT ||
			direction.To.Type != pb.EPointKeyType_POINT_KEY_TYPE_SETTLEMENT {
			a.logger.Errorf("%s: bad direction. Skip it", errorMessage)
			continue
		}
		fromSettlement, ok := a.raspRepo.GetSettlement(int32(direction.From.Id))
		if !ok {
			a.logger.Infof("%s: unknown settlement id=%d. Skip it", errorMessage, direction.From.Id)
			continue
		}
		toSettlement, ok := a.raspRepo.GetSettlement(int32(direction.To.Id))
		if !ok {
			a.logger.Infof("%s: unknown settlement id=%d. Skip it", errorMessage, direction.To.Id)
			continue
		}

		calendarItems, err := a.Calendar(
			direction.From, direction.To,
			utils.ConvertTimeToProtoDate(minDate), utils.ConvertTimeToProtoDate(maxDate),
			pb.ERequestSource_SRS_POPULAR, ctx,
		)
		if err != nil {
			a.logger.Errorf("%s: %s", errorMessage, err.Error())
			continue
		}
		var minPrices []*tpb.TPrice
		for _, calendarItem := range calendarItems {
			for _, price := range calendarItem.MinPrices {
				minPrices = updatePrices(minPrices, price)
			}
		}
		if len(minPrices) == 0 {
			continue
		}
		response.Directions[counter] = &ipb.TPopularDirectionResponse{
			From: &rpb.TSettlement{
				Id:           fromSettlement.Id,
				TitleDefault: fromSettlement.TitleDefault,
				Longitude:    fromSettlement.Longitude,
				Latitude:     fromSettlement.Latitude,
				Slug:         fromSettlement.Slug,
				Title:        fromSettlement.Title,
				CountryId:    fromSettlement.CountryId,
			},
			To: &rpb.TSettlement{
				Id:           toSettlement.Id,
				TitleDefault: toSettlement.TitleDefault,
				Longitude:    toSettlement.Longitude,
				Latitude:     toSettlement.Latitude,
				Slug:         toSettlement.Slug,
				Title:        toSettlement.Title,
				CountryId:    fromSettlement.CountryId,
			},
			Prices: minPrices,
			Weight: direction.Weight,
		}
		counter++
	}
	if counter < limit {
		response.Directions = response.Directions[:counter]
	}
	return response
}

func (a *App) GetPopularDirections(limit int, ctx context.Context) *ipb.TPopularDirectionsResponse {
	return a.GetPopularDirectionsFrom(nil, limit, ctx)
}
