package app

import (
	"context"
	"errors"
	"fmt"
	"time"

	"a.yandex-team.ru/library/go/core/metrics"
	tpb "a.yandex-team.ru/travel/proto"
	"github.com/golang/protobuf/proto"

	"a.yandex-team.ru/travel/buses/backend/internal/common/connector"
	"a.yandex-team.ru/travel/buses/backend/internal/common/dict"
	"a.yandex-team.ru/travel/buses/backend/internal/common/utils"
	pb "a.yandex-team.ru/travel/buses/backend/proto"
)

var errRideNotFound = errors.New("no such ride-id in ride index")

type RideResult struct {
	QueryFrom      *pb.TPointKey
	QueryTo        *pb.TPointKey
	Ride           *pb.TRide
	BookParams     *pb.TBookParams
	SubstitutionID string
}

func (a *App) Ride(rideID string, allowSubstitution bool, ctx context.Context) (RideResult, *StatusWithMessage) {
	const funcName = "App.Ride"

	rideResult, rideGettingStatus := a.ride(rideID, allowSubstitution, ctx)

	if rideGettingStatus.RideStatus.Ok() {
		if rideResult.Ride.GetPrice() == nil {
			// TODO: move to filters
			return RideResult{}, NewStatusWithMessage(pb.EStatus_STATUS_INTERNAL_ERROR, "Ride has no price")
		}
		rideWithFee, bookParamsWithFee, err := a.billingDict.WithYandexFee(rideResult.Ride, rideResult.BookParams)
		if err != nil {
			return RideResult{}, NewStatusWithMessage(pb.EStatus_STATUS_INTERNAL_ERROR, err.Error())
		}
		rideResult.Ride = rideWithFee
		rideResult.BookParams = bookParamsWithFee
	}

	supplierName := ""
	if supplierID, _, err := utils.ParseRideID(rideID); err == nil {
		if supplier, err := dict.GetSupplier(supplierID); err == nil {
			supplierName = supplier.Name
		}
	}

	a.addRideMetrics(rideGettingStatus, supplierName)

	status := rideGettingStatus.RideStatus
	if !rideGettingStatus.RideStatus.Ok() {
		if rideGettingStatus.RideStatus.Status == pb.EStatus_STATUS_INTERNAL_ERROR {
			a.logger.Errorf("%s: rideID=%s: %s", funcName, rideID, rideGettingStatus.RideStatus.Message)
		} else {
			a.logger.Infof("%s: rideID=%s: RideStatus=%s, SubstitutionStatus=%s", funcName, rideID,
				rideGettingStatus.RideStatus.String(), rideGettingStatus.SubstitutionStatus.String())
		}
		if rideGettingStatus.SubstitutionStatus != nil &&
			(rideGettingStatus.SubstitutionStatus.Status == pb.EStatus_STATUS_RIDE_SUBSTITUTION ||
				rideGettingStatus.SubstitutionStatus.Status == pb.EStatus_STATUS_RIDE_SUBSTITUTION_FROM_OTHER) {
			a.logger.Infof("%s: changed to %s (rideID = %s)", funcName, rideResult.SubstitutionID, rideID)
			status = rideGettingStatus.SubstitutionStatus
		}
	}

	return rideResult, status
}

type rideGettingStatusWithMessage struct {
	RideStatus         *StatusWithMessage
	SubstitutionStatus *StatusWithMessage
}

func (a *App) ride(rideID string, allowSubstitution bool, ctx context.Context) (RideResult, *rideGettingStatusWithMessage) {
	if _, _, err := utils.ParseRideID(rideID); err != nil {
		return RideResult{}, &rideGettingStatusWithMessage{
			RideStatus: NewStatusWithMessage(pb.EStatus_STATUS_BAD_REQUEST, err.Error()),
		}
	}

	cachedRide, query, err := a.findRideInCache(rideID)
	if err != nil {
		if errors.Is(err, errRideNotFound) {
			return RideResult{}, &rideGettingStatusWithMessage{
				RideStatus: NewStatusWithMessage(pb.EStatus_STATUS_NOT_FOUND, err.Error()),
			}
		}
		return RideResult{}, &rideGettingStatusWithMessage{
			RideStatus: NewStatusWithMessage(pb.EStatus_STATUS_INTERNAL_ERROR, err.Error()),
		}
	}

	result := RideResult{
		QueryFrom: query.from,
		QueryTo:   query.to,
		Ride:      cachedRide,
	}
	ride, status := a.findActualRide(cachedRide, query, ctx)
	if status.RideStatus.Ok() {
		bookParams, substitute, status := a.bookParams(ride)
		if status.Ok() {
			result.Ride = ride
			result.BookParams = bookParams
			return result, &rideGettingStatusWithMessage{
				RideStatus: status,
			}
		}
		a.bannedRidesRule.Register(ride)
		if !allowSubstitution {
			return result, &rideGettingStatusWithMessage{
				RideStatus:         status,
				SubstitutionStatus: NewStatusWithMessage(pb.EStatus_STATUS_RIDE_SUBSTITUTION_DISALLOWED_BY_ARGUMENT, ""),
			}
		}
		if !substitute {
			return result, &rideGettingStatusWithMessage{
				RideStatus:         status,
				SubstitutionStatus: NewStatusWithMessage(pb.EStatus_STATUS_RIDE_SUBSTITUTION_DISALLOWED_BY_BOOK_PARAMS, ""),
			}
		}
		return a.ride(rideID, allowSubstitution, ctx)
	} else if status.SubstitutionStatus == nil || status.SubstitutionStatus.Status == pb.EStatus_STATUS_RIDE_NO_SUBSTITUTION {
		otherSuppliersRides, _ := a.Search(
			query.from, query.to, query.date, true,
			pb.ERequestSource_SRS_SYNC_SEARCH, ctx)

		ride, status = a.chooseRide(cachedRide, otherSuppliersRides)
		if status.SubstitutionStatus == nil || status.SubstitutionStatus.Status == pb.EStatus_STATUS_RIDE_NO_SUBSTITUTION {
			return result, status
		}
	}

	priceChange := (ride.Price.Amount - cachedRide.Price.Amount) / 100
	if priceChange < 0 {
		priceChange = -priceChange
	}
	a.appMetrics.GetOrCreateHistogram(
		"rides", nil, "substitution_price_diff",
		metrics.NewDurationBuckets(0, 50*time.Second, 200*time.Second, 1000*time.Second, 10000*time.Second),
	).RecordDuration(time.Duration(priceChange) * time.Second)

	result.SubstitutionID = ride.Id
	return result, status
}

type searchQuery struct {
	supplierID uint32
	from       *pb.TPointKey
	to         *pb.TPointKey
	date       *tpb.TDate
}

func (a *App) findRideInCache(rideID string) (*pb.TRide, searchQuery, error) {
	searchKey, ok := a.searchCache.GetFirstSearchKeyByRideID(rideID)
	if !ok {
		return nil, searchQuery{}, errRideNotFound
	}
	query := searchQuery{
		supplierID: searchKey.SupplierID,
		from:       &pb.TPointKey{Type: searchKey.FromType, Id: searchKey.FromID},
		to:         &pb.TPointKey{Type: searchKey.ToType, Id: searchKey.ToID},
		date:       &tpb.TDate{Year: searchKey.DateYear, Month: searchKey.DateMonth, Day: searchKey.DateDay},
	}
	searchRecord, ok := a.searchCache.Get(searchKey)
	if !ok {
		return nil, query, errors.New("inconsistent cache: no record")
	}
	for _, r := range searchRecord.Rides {
		if r.Id == rideID {
			return proto.Clone(r).(*pb.TRide), query, nil
		}
	}
	return nil, query, errors.New("inconsistent cache: no ride")
}

func (a *App) findActualRide(
	cachedRide *pb.TRide, query searchQuery, ctx context.Context,
) (*pb.TRide, *rideGettingStatusWithMessage) {
	actualRides, err := a.syncSearch(query.supplierID, query.from, query.to, query.date, ctx)
	if err != nil {
		return nil, &rideGettingStatusWithMessage{
			RideStatus: NewStatusWithMessage(pb.EStatus_STATUS_RIDE_SEARCH_FAILS, err.Error()),
		}
	}
	actualRides = a.ridesFilter.Filter(query.from, query.to, actualRides, a.logger)
	actualRides = a.enrichRides(actualRides)
	return a.chooseRide(cachedRide, actualRides)
}

func (a *App) chooseRide(cachedRide *pb.TRide, actualRides Rides) (*pb.TRide, *rideGettingStatusWithMessage) {
	for _, ride := range actualRides {
		if ride.Id == cachedRide.Id {
			return ride, &rideGettingStatusWithMessage{
				RideStatus: NewStatusWithMessage(pb.EStatus_STATUS_OK, ""),
			}
		}
	}
	rideStatus := pb.EStatus_STATUS_RIDE_DISAPPEARED
	substitutionStatus := pb.EStatus_STATUS_RIDE_NO_SUBSTITUTION
	for _, ride := range actualRides {
		if proto.Equal(ride.From, cachedRide.From) &&
			proto.Equal(ride.To, cachedRide.To) &&
			ride.DepartureTime == cachedRide.DepartureTime {
			substitutionStatus = pb.EStatus_STATUS_RIDE_SUBSTITUTION
			if ride.SupplierId != cachedRide.SupplierId {
				substitutionStatus = pb.EStatus_STATUS_RIDE_SUBSTITUTION_FROM_OTHER
			}
			return ride, &rideGettingStatusWithMessage{
				RideStatus: NewStatusWithMessage(rideStatus, ""),
				SubstitutionStatus: NewStatusWithMessage(substitutionStatus,
					fmt.Sprintf("original ride changed: %s -> %s", cachedRide.Id, ride.Id)),
			}
		}
	}
	return nil, &rideGettingStatusWithMessage{
		RideStatus:         NewStatusWithMessage(rideStatus, ""),
		SubstitutionStatus: NewStatusWithMessage(substitutionStatus, ""),
	}
}

func (a *App) addRideMetrics(rideGettingStatus *rideGettingStatusWithMessage, supplierName string) {
	tags := make(map[string]string)
	if rideGettingStatus.RideStatus != nil {
		tags["ride_status"] = rideGettingStatus.RideStatus.Status.String()
	}
	if rideGettingStatus.SubstitutionStatus != nil {
		tags["substitution_status"] = rideGettingStatus.SubstitutionStatus.Status.String()
	}
	if supplierName != "" {
		tags["supplier"] = supplierName
	}
	a.appMetrics.GetOrCreateCounter("rides", tags, "get").Inc()
}

func (a *App) enrichRides(rides []*pb.TRide) []*pb.TRide {
	const logMessage = "App.enrichRides"

	enrichedRides := make([]*pb.TRide, len(rides))
	i := 0
	for _, ride := range rides {
		rideCopy := proto.Clone(ride).(*pb.TRide)

		if ride.From == nil || ride.To == nil {
			a.logger.Errorf("%s: bad ride data for rideId=%s", logMessage, ride.Id)
		} else if rideCopy.ArrivalTime != 0 {
			timeLocationFrom, fromErr := utils.GetPointKeyTimeZone(a.raspRepo, ride.From)
			timeLocationTo, toErr := utils.GetPointKeyTimeZone(a.raspRepo, ride.To)
			if fromErr != nil || toErr != nil {
				if fromErr != nil {
					a.logger.Infof("%s: can not get timezone for %s: %s", logMessage, ride.From.String(), fromErr.Error())
				}
				if toErr != nil {
					a.logger.Infof("%s: can not get timezone for %s: %s", logMessage, ride.To.String(), toErr.Error())
				}
				timeLocationFrom = time.UTC
				timeLocationTo = time.UTC
			}
			departureTime := utils.ConvertRideSecondsToTime(ride.DepartureTime, timeLocationFrom)
			arrivalTime := utils.ConvertRideSecondsToTime(ride.ArrivalTime, timeLocationTo)

			rideCopy.Duration = int64(arrivalTime.Sub(departureTime).Seconds())
		}

		enrichedRides[i] = rideCopy
		i++
	}
	if i < len(rides) {
		enrichedRides = enrichedRides[:i]
	}
	return enrichedRides
}

func (a *App) bookParams(ride *pb.TRide) (_bookParams *pb.TBookParams, _substitute bool, statusWithMessage *StatusWithMessage) {
	const (
		funcName        = "App.bookParams"
		doSubstitute    = true
		doNotSubstitute = false
	)

	defer func() {
		if !statusWithMessage.Ok() {
			statusWithMessage = NewStatusWithMessage(
				statusWithMessage.Status, fmt.Sprintf("%s: %v", funcName, statusWithMessage.Message))
		}
	}()

	conn, err := connector.NewClient(&a.cfg.Connector, ride.SupplierId, a.logger)
	if err != nil {
		return nil, doNotSubstitute, NewStatusWithMessage(pb.EStatus_STATUS_INTERNAL_ERROR, err.Error())
	}

	rideID := ride.Id
	bookParams, rideRefinement, _, err := conn.GetBookParams(rideID)
	if err != nil {
		var connectorErr connector.ErrWithMetadata
		return nil, errors.As(err, &connectorErr), NewStatusWithMessage(pb.EStatus_STATUS_RIDE_BOOK_PARAMS_ERROR, err.Error())
	}

	if err := refineRide(ride, rideRefinement); err != nil {
		return nil, doSubstitute, NewStatusWithMessage(pb.EStatus_STATUS_RIDE_REFINEMENT_ERROR, err.Error())
	}

	return bookParams, doNotSubstitute, NewStatusWithMessage(pb.EStatus_STATUS_OK, "")
}

func refineRide(ride *pb.TRide, rideRefinement *pb.TRideRefinement) error {
	if rideRefinement.DepartureTime != 0 && rideRefinement.DepartureTime != ride.DepartureTime {
		return fmt.Errorf("departure time is changed from %d to %d for %s", ride.DepartureTime, rideRefinement.DepartureTime, ride.Id)
	}
	if rideRefinement.ArrivalTime != 0 {
		ride.ArrivalTime = rideRefinement.ArrivalTime
	}
	if rideRefinement.From != nil {
		ride.From = rideRefinement.From
	}
	if rideRefinement.FromDesc != "" {
		ride.FromDesc = rideRefinement.FromDesc
	}
	if rideRefinement.To != nil {
		ride.To = rideRefinement.To
	}
	if rideRefinement.ToDesc != "" {
		ride.ToDesc = rideRefinement.ToDesc
	}
	if rideRefinement.Price != nil && rideRefinement.Price.Amount != 0 {
		ride.Price.Amount = rideRefinement.Price.Amount
	}
	if rideRefinement.Fee != nil && rideRefinement.Fee.Amount != 0 {
		ride.Fee.Amount = rideRefinement.Fee.Amount
	}
	if rideRefinement.RefundConditions != "" {
		ride.RefundConditions = rideRefinement.RefundConditions
	}
	return nil
}
