package app

import (
	"context"
	"fmt"
	"sort"
	"time"

	tpb "a.yandex-team.ru/travel/proto"

	"a.yandex-team.ru/travel/buses/backend/internal/api/cache"
	"a.yandex-team.ru/travel/buses/backend/internal/common/dict"
	ipb "a.yandex-team.ru/travel/buses/backend/internal/common/proto"
	"a.yandex-team.ru/travel/buses/backend/internal/common/utils"
	pb "a.yandex-team.ru/travel/buses/backend/proto"
	wpb "a.yandex-team.ru/travel/buses/backend/proto/worker"
)

type Rides []*pb.TRide

func (a *App) Search(
	from *pb.TPointKey, to *pb.TPointKey, date *tpb.TDate, deduplicate bool,
	source pb.ERequestSource, ctx context.Context,
) (Rides, bool) {
	a.registerSearchQuery(from, to, source)

	var rides, ready = a.findRides(from, to, date, deduplicate, source, ctx)
	if ready {
		a.searchMetricHistogram.RecordValue(float64(len(rides)))
	}

	return rides, ready
}

func (a *App) SearchRange(
	from *pb.TPointKey, to *pb.TPointKey, minDate *tpb.TDate, days uint32,
	source pb.ERequestSource, ctx context.Context,
) (*tpb.TDate, Rides, bool) {
	a.registerSearchQuery(from, to, source)

	var (
		minTime    = utils.ConvertProtoDateToTime(minDate)
		rangeReady = true
	)
	for day := uint32(0); day < days; day++ {
		var (
			searchDate   = utils.ConvertTimeToProtoDate(minTime.AddDate(0, 0, int(day)))
			rides, ready = a.findRides(from, to, searchDate, true, source, ctx)
		)
		if len(rides) > 0 {
			return searchDate, rides, ready
		}
		rangeReady = rangeReady && ready
	}

	return nil, nil, rangeReady
}

func (a *App) registerSearchQuery(from *pb.TPointKey, to *pb.TPointKey, source pb.ERequestSource) {
	if source == pb.ERequestSource_SRS_SEARCH || source == pb.ERequestSource_SRS_RASP_SEARCH {
		a.popularDirections.Register(from, to)
	}
	if a.isMaster {
		a.searchCacheWarmer.RegisterQuery(from, to, source)
	}
}

func (a *App) findRides(
	from *pb.TPointKey, to *pb.TPointKey, date *tpb.TDate, deduplicate bool,
	source pb.ERequestSource, _ context.Context,
) (Rides, bool) {
	const logMessage = "App.findRides"

	var supplierIDs = a.cfg.Suppliers
	if len(supplierIDs) == 0 {
		supplierIDs = dict.GetSuppliersList()
	}

	var (
		ready = true
		rides = make([]*pb.TRide, 0, 10)
	)
	for _, supplierID := range supplierIDs {
		if !a.checkSegmentAndSchedule(supplierID, from, to) {
			continue
		}

		var (
			searchKey = cache.NewSearchKey(supplierID, from, to, date)
			nowTS     = time.Now().Unix()
		)
		if searchRecord, ok := a.searchCache.Get(searchKey); ok {
			var searchRecordAge = time.Duration(nowTS-searchRecord.CreatedAt) * time.Second
			switch searchRecord.Status {
			case ipb.ECacheRecordStatus_CACHE_RECORD_STATUS_OK:
				rides = append(rides, searchRecord.Rides...)
				continue
			case ipb.ECacheRecordStatus_CACHE_RECORD_STATUS_REQUESTED:
				if searchRecordAge < a.cfg.FrontSearchTimeout {
					ready = false
					continue
				}
				a.searchCache.Set(searchKey, &ipb.TSearchCacheRecord{
					Status:    ipb.ECacheRecordStatus_CACHE_RECORD_STATUS_FAILED,
					CreatedAt: nowTS,
				})
				continue
			case ipb.ECacheRecordStatus_CACHE_RECORD_STATUS_FAILED:
				if searchRecordAge < a.cfg.SearchFailedTimeout {
					continue
				}
			}
		}
		ready = false

		if source != pb.ERequestSource_SRS_CALENDAR && source != pb.ERequestSource_SRS_POPULAR {
			a.searchCache.Set(searchKey, &ipb.TSearchCacheRecord{
				Status:    ipb.ECacheRecordStatus_CACHE_RECORD_STATUS_REQUESTED,
				CreatedAt: nowTS,
			})

			go func(sID uint32) {
				err := a.scheduleSearch(sID, from, to, date,
					getRequestSourceTryNoCache(source), getRequestSourcePriority(source))
				if err != nil {
					a.searchCache.Set(searchKey, &ipb.TSearchCacheRecord{
						Status:    ipb.ECacheRecordStatus_CACHE_RECORD_STATUS_FAILED,
						CreatedAt: nowTS,
					})
					a.logger.Errorf("%s: %s", logMessage, err.Error())
				}
			}(supplierID)
		}
	}

	rides = a.ridesFilter.Filter(from, to, rides, a.logger)
	rides = a.billingDict.RidesWithYandexFee(rides, a.logger)
	rides = a.enrichRides(rides)
	if deduplicate {
		rides = a.deduplicate(rides)
	}

	return rides, ready
}

func (a *App) syncSearch(
	supplierID uint32, from *pb.TPointKey, to *pb.TPointKey, date *tpb.TDate, _ context.Context,
) (Rides, error) {
	const (
		errorMessage = "App.syncSearch"
		source       = pb.ERequestSource_SRS_SYNC_SEARCH
	)

	searchResult := &wpb.TSearchResult{}
	searchChannel := a.searchConsumer.NewChannel()
	defer func() { _ = a.searchConsumer.CloseChannel(searchChannel) }()

	searchKey := cache.NewSearchKey(supplierID, from, to, date)
	a.searchCache.Set(searchKey, &ipb.TSearchCacheRecord{
		Status:    ipb.ECacheRecordStatus_CACHE_RECORD_STATUS_REQUESTED,
		CreatedAt: time.Now().Unix(),
	})

	err := a.scheduleSearch(
		supplierID, from, to, date,
		getRequestSourceTryNoCache(source), getRequestSourcePriority(source),
	)
	if err != nil {
		return nil, err
	}

	deadline := time.Now().Add(a.cfg.FrontSearchTimeout)
	for {
		err := searchChannel.ReadWithDeadline(searchResult, deadline)
		if err != nil {
			return nil, fmt.Errorf("%s: %w", errorMessage, err)
		}
		if searchKey != cache.NewSearchKey(
			searchResult.Request.SupplierId, searchResult.Request.From,
			searchResult.Request.To, searchResult.Request.Date) {
			continue
		}
		if searchResult.Request.TryNoCache != getRequestSourceTryNoCache(source) {
			continue
		}
		if searchResult.Header.Code != tpb.EErrorCode_EC_OK {
			return nil, fmt.Errorf("%s: error message received with code=%s", errorMessage, searchResult.Header.Code)
		}
		return searchResult.Rides, nil
	}
}

func (a *App) scheduleSearch(
	supplierID uint32, from *pb.TPointKey, to *pb.TPointKey,
	date *tpb.TDate, tryNoCache bool, priority wpb.ERequestPriority,
) error {
	const logMessage = "App.scheduleSearch"
	searchResponse, err := a.workerServiceClient.Search(a.ctx, &wpb.TSearchRequest{
		Header: &wpb.TRequestHeader{
			Priority: priority,
		},
		SupplierId: supplierID,
		From:       from,
		To:         to,
		Date:       date,
		TryNoCache: tryNoCache,
	})
	if err != nil {
		return fmt.Errorf("%s: error while GRPC call: %w", logMessage, err)
	}
	if searchResponse.Header.Code != tpb.EErrorCode_EC_OK {
		return fmt.Errorf("%s: search scheduling error: %v", logMessage, searchResponse.Header.Error)
	}
	return nil
}

type RidesSorter struct {
	rides   Rides
	billing *BillingData
}

func (r RidesSorter) Len() int {
	return len(r.rides)
}

func (r RidesSorter) Swap(i, j int) {
	r.rides[i], r.rides[j] = r.rides[j], r.rides[i]
}

func (r RidesSorter) Less(i, j int) bool {
	const logMessage = "RidesSorter.Less"
	ride1, ride2 := r.rides[i], r.rides[j]
	// compare order: departure time, fromID, freeSeats, price, margin, partner priority
	if ride1.DepartureTime < ride2.DepartureTime {
		return true
	}
	if ride1.DepartureTime > ride2.DepartureTime {
		return false
	}
	from1Str, _ := utils.DumpPointKey(ride1.From)
	from2Str, _ := utils.DumpPointKey(ride2.From)
	if from1Str < from2Str {
		return false
	}
	if from1Str > from2Str {
		return true
	}
	var seats1, seats2 int
	if ride1.FreeSeats != 0 {
		seats1 = 1
	}
	if ride2.FreeSeats != 0 {
		seats2 = 1
	}
	if seats1 < seats2 {
		return false
	} else if seats1 > seats2 {
		return true
	}
	// TODO: no currency checks
	if ride1.Price.Amount < ride2.Price.Amount {
		return true
	} else if ride1.Price.Amount > ride2.Price.Amount {
		return false
	}
	supplier1, err1 := dict.GetSupplier(ride1.SupplierId)
	if err1 != nil {
		panic(fmt.Sprintf("%s: unknown supplier=%d", logMessage, ride1.SupplierId))
	}
	supplier2, err2 := dict.GetSupplier(ride2.SupplierId)
	if err2 != nil {
		panic(fmt.Sprintf("%s: unknown supplier=%d", logMessage, ride2.SupplierId))
	}
	revenue1, revenue2 := r.billing.GetRevenue(supplier1.Name), r.billing.GetRevenue(supplier2.Name)
	if revenue1 < revenue2 {
		return false
	} else if revenue1 > revenue2 {
		return true
	}
	return supplier1.Priority < supplier2.Priority
}

func (a *App) deduplicate(rides Rides) Rides {
	if len(rides) <= 1 {
		return rides
	}
	ridesToSort := RidesSorter{
		rides:   rides,
		billing: a.billingDict,
	}
	deduplicatedRides := make(Rides, 0, len(rides))
	// sort all rides by all groups and ratings
	sort.Sort(ridesToSort)

	firstInGroup := true
	curDepartGroup := ridesToSort.rides[0].DepartureTime
	curFromGroup, _ := utils.DumpPointKey(ridesToSort.rides[0].From)

	for _, ride := range ridesToSort.rides {
		fromStr, _ := utils.DumpPointKey(ride.From)
		if ride.DepartureTime != curDepartGroup {
			curDepartGroup = ride.DepartureTime
			curFromGroup = fromStr
			firstInGroup = true
		}
		if firstInGroup {
			deduplicatedRides = append(deduplicatedRides, ride)
			firstInGroup = false
			continue
		}
		if fromStr == curFromGroup {
			continue
		}
		if ride.From.Type == pb.EPointKeyType_POINT_KEY_TYPE_STATION {
			deduplicatedRides = append(deduplicatedRides, ride)
			curFromGroup = fromStr
			continue
		}
	}

	return deduplicatedRides
}

func (a *App) searchConsumerLoop() {
	const errorMessage = "App.searchConsumerLoop"
	var searchResult = &wpb.TSearchResult{}
	searchChannel := a.searchConsumer.NewChannel()
	defer func() { _ = a.searchConsumer.CloseChannel(searchChannel) }()
	for {
		if err := searchChannel.Read(searchResult); err != nil {
			a.logger.Errorf("%s: can not read search result: %s", errorMessage, err.Error())
			select {
			case <-time.After(time.Second):
				continue
			case <-a.ctx.Done():
				a.logger.Infof("%s: stopped", errorMessage)
				return
			}
		}
		if searchResult.Request.From == nil || searchResult.Request.To == nil || searchResult.Request.Date == nil {
			a.logger.Errorf("%s: searchResult.Request has empty fields", errorMessage)
			continue
		}

		searchKey := cache.NewSearchKey(
			searchResult.Request.SupplierId,
			searchResult.Request.From,
			searchResult.Request.To,
			searchResult.Request.Date,
		)
		curTime := time.Now()
		var cacheRecord *ipb.TSearchCacheRecord
		if searchResult.Header.Code != tpb.EErrorCode_EC_OK {
			if time.Unix(searchResult.Header.Timestamp, 0).Add(a.cfg.SearchFailedTimeout).Before(curTime) {
				continue
			}
			if searchResult.Header.Error == nil {
				a.logger.Infof("Bad search result: %s", searchResult.Header.Code)
			} else {
				a.logger.Infof("Bad search result: %s: %s", searchResult.Header.Error.Code,
					searchResult.Header.Error.Message)
			}
			cacheRecord = &ipb.TSearchCacheRecord{
				Status:    ipb.ECacheRecordStatus_CACHE_RECORD_STATUS_FAILED,
				CreatedAt: searchResult.Header.Timestamp,
			}
		} else {
			if time.Unix(searchResult.Header.Timestamp, 0).Add(a.cfg.SearchCacheTTL).Before(curTime) {
				continue
			}
			cacheRecord = &ipb.TSearchCacheRecord{
				Status:    ipb.ECacheRecordStatus_CACHE_RECORD_STATUS_OK,
				Rides:     searchResult.Rides,
				CreatedAt: searchResult.Header.Timestamp,
			}
		}
		a.searchCache.Set(searchKey, cacheRecord)
	}
}

func getRequestSourcePriority(source pb.ERequestSource) wpb.ERequestPriority {
	switch source {
	case pb.ERequestSource_SRS_SYNC_SEARCH:
		return wpb.ERequestPriority_REQUEST_PRIORITY_HIGH
	case pb.ERequestSource_SRS_SEARCH, pb.ERequestSource_SRS_RASP_SEARCH:
		return wpb.ERequestPriority_REQUEST_PRIORITY_NORMAL
	}
	return wpb.ERequestPriority_REQUEST_PRIORITY_LOW
}

func getRequestSourceTryNoCache(source pb.ERequestSource) bool {
	return source == pb.ERequestSource_SRS_SYNC_SEARCH
}
