package app

import (
	"context"
	"time"

	tpb "a.yandex-team.ru/travel/proto"

	"a.yandex-team.ru/travel/buses/backend/internal/common/logging"
	ipb "a.yandex-team.ru/travel/buses/backend/internal/common/proto"
	pb "a.yandex-team.ru/travel/buses/backend/proto"
	wpb "a.yandex-team.ru/travel/buses/backend/proto/worker"
)

func (a *App) checkSegmentAndSchedule(supplierID uint32, from *pb.TPointKey, to *pb.TPointKey) bool {

	status, timestamp := a.segmentsCache.GetStatus(supplierID)
	age := time.Since(timestamp)
	if status == ipb.ECacheRecordStatus_CACHE_RECORD_STATUS_MISSED ||
		status == ipb.ECacheRecordStatus_CACHE_RECORD_STATUS_OK && age > a.cfg.SegmentsCacheTTL ||
		status == ipb.ECacheRecordStatus_CACHE_RECORD_STATUS_REQUESTED && age > a.cfg.SegmentsFailedTimeout ||
		status == ipb.ECacheRecordStatus_CACHE_RECORD_STATUS_FAILED && age > a.cfg.SegmentsFailedTimeout {
		a.segmentsCache.SetStatus(supplierID, ipb.ECacheRecordStatus_CACHE_RECORD_STATUS_REQUESTED, time.Now())
		go a.scheduleSegments(supplierID, a.ctx)
	}

	return a.segmentsCache.Has(supplierID, from, to)
}

func (a *App) scheduleSegments(sID uint32, ctx context.Context) {
	const logMessage = "App.scheduleSegments"

	segmentsResponse, err := a.workerServiceClient.Segments(ctx, &wpb.TSegmentsRequest{
		Header:     &wpb.TRequestHeader{},
		SupplierId: sID,
	})
	if err != nil {
		a.logger.Errorf("%s: error while GRPC call: %s", logMessage, err.Error())
		return
	}
	if segmentsResponse.Header.Code != tpb.EErrorCode_EC_OK {
		a.logger.Errorf("%s: scheduling error: %v", logMessage, segmentsResponse.Header.Error)
		return
	}
}

func (a *App) segmentsConsumerLoop() {
	const logMessage = "App.segmentsConsumerLoop"
	var result = wpb.TSegmentsResult{}
	segmentsChannel := a.segmentsConsumer.NewChannel()
	defer func() { _ = a.segmentsConsumer.CloseChannel(segmentsChannel) }()
	for {
		if err := segmentsChannel.Read(&result); err != nil {
			a.logger.Errorf("%s: can not get segments result: %s", logMessage, err.Error())
			select {
			case <-time.After(time.Second):
				continue
			case <-a.ctx.Done():
				a.logger.Infof("%s: stopped", logMessage)
				return
			}
		}

		timestamp := time.Unix(result.Header.Timestamp, 0)
		supplierID := result.Request.SupplierId

		if result.Header.Code == tpb.EErrorCode_EC_OK {
			a.segmentsCache.SetSegments(supplierID, result.Segments, timestamp)
			logging.WithSupplierContext(a.logger, supplierID).Infof(
				"%s: set segments: %d records", logMessage, len(result.Segments))
		} else {
			a.segmentsCache.SetStatus(supplierID, ipb.ECacheRecordStatus_CACHE_RECORD_STATUS_FAILED, timestamp)
			logging.WithSupplierContext(a.logger, supplierID).Infof(
				"%s: bad segments result: %s: %v", logMessage, result.Header.Code.String(), result.Header.Error)
		}
	}
}
