package app

import (
	"context"
	"fmt"
	"net/url"
	"time"

	"github.com/go-chi/chi/v5/middleware"

	tpb "a.yandex-team.ru/travel/proto"
	"a.yandex-team.ru/travel/proto/dicts/rasp"

	ipb "a.yandex-team.ru/travel/buses/backend/internal/common/proto"
	"a.yandex-team.ru/travel/buses/backend/internal/common/utils"
	pb "a.yandex-team.ru/travel/buses/backend/proto"
)

const wizardSearchDays = 7

func (a *App) Wizard(
	fromGeoID int32, from *pb.TPointKey, toGeoID int32, to *pb.TPointKey,
	date *tpb.TDate, tld string, mainReqID string, ctx context.Context,
) (*ipb.TWizardResponse, *StatusWithMessage) {
	const logMessage = "App.Wizard"
	const ruTLD = "ru"

	if tld != ruTLD {
		return nil, NewStatusWithMessage(pb.EStatus_STATUS_NOT_FOUND,
			fmt.Sprintf("%s: prices are allowed only for tld=%s", logMessage, ruTLD))
	}

	from, fromSettlement, status := a.getPointKeyAndSettlement(fromGeoID, from)
	if !status.Ok() {
		return nil, status
	}
	to, toSettlement, status := a.getPointKeyAndSettlement(toGeoID, to)
	if !status.Ok() {
		return nil, status
	}

	timezone, err := utils.GetPointKeyTimeZone(a.raspRepo, from)
	today := time.Now()
	if err != nil {
		a.logger.Infof("%s: can not get timezone for %v: %s", logMessage, from, err.Error())
	} else {
		today = today.In(timezone)
	}
	tomorrow := today.AddDate(0, 0, 1)
	searchDate := date
	var (
		rides Rides
		ready bool
	)
	if searchDate != nil {
		rides, ready = a.Search(
			from, to, searchDate, true,
			pb.ERequestSource_SRS_WIZARD, ctx,
		)
	} else {
		searchDate, rides, ready = a.SearchRange(
			from, to, utils.ConvertTimeToProtoDate(tomorrow), wizardSearchDays,
			pb.ERequestSource_SRS_WIZARD, ctx,
		)
	}

	if len(rides) == 0 {
		if ready {
			return nil, NewStatusWithMessage(pb.EStatus_STATUS_NOT_FOUND, fmt.Sprintf("%s: no rides", logMessage))
		}
		return nil, NewStatusWithMessage(pb.EStatus_STATUS_NOT_READY, fmt.Sprintf("%s: no rides", logMessage))
	}

	meanTime := 0
	var minPrices []*tpb.TPrice
	additives := make([]*ipb.TRideAdditive, len(rides))
	fromStr, _ := utils.DumpPointKey(from)
	toStr, _ := utils.DumpPointKey(to)
	dateStr := utils.DumpDate(searchDate)

	params := url.Values{}
	reqID := middleware.GetReqID(ctx)
	if reqID != "" {
		params.Set("req_id", reqID)
		params.Set("wizardReqId", reqID)
	}
	if mainReqID != "" {
		params.Set("wizardReqId", mainReqID)
	}

	wizParams := params
	wizParams.Set("utm_source", "common_wizard")
	wizParams.Set("utm_campaign", "new")

	wizBuyParams := wizParams
	wizBuyParams.Set("utm_medium", "buy_button")

	wizSnippetParams := wizParams
	wizSnippetParams.Set("utm_medium", "snippet")
	wizSnippetGreenURLParams := wizSnippetParams
	wizSnippetGreenURLParams.Set("utm_content", "greenURL")
	wizSnippetSerpURLParams := wizSnippetParams
	wizSnippetSerpURLParams.Set("utm_content", "serpURL")
	wizSnippetSerpURLParams.Set("date", utils.DumpDate(searchDate))
	wizSnippetUpcomingParams := wizSnippetParams
	wizSnippetUpcomingParams.Set("utm_content", "upcoming")
	wizSnippetUpcomingParams.Set("date", today.Format(utils.DateLayout))
	wizSnippetTomorrowParams := wizSnippetParams
	wizSnippetTomorrowParams.Set("utm_content", "tomorrow")
	wizSnippetTomorrowParams.Set("date", tomorrow.Format(utils.DateLayout))

	raspBuyParams := params
	raspBuyParams.Set("utm_source", "rasp")
	raspBuyParams.Set("utm_medium", "buy_button")

	for i, ride := range rides {
		meanTime += int(ride.Duration)
		minPrices = updatePrices(minPrices, &tpb.TPrice{
			Currency: ride.Price.Currency,
			Amount:   ride.Price.Amount,
		})
		additives[i] = &ipb.TRideAdditive{
			BookUrlOrig: utils.BookURL(tld, fromStr, toStr, dateStr, ride.Id, raspBuyParams),
			BookUrl:     utils.BookURL(tld, fromStr, toStr, dateStr, ride.Id, wizBuyParams),
		}
	}
	meanTime /= len(rides)
	if len(minPrices) == 0 {
		return nil, NewStatusWithMessage(pb.EStatus_STATUS_INTERNAL_ERROR,
			fmt.Sprintf("%s: no prices for not empty search result", logMessage))
	}

	frontURL, err := url.Parse(fmt.Sprintf(
		"%s/%s/%s/",
		fmt.Sprintf(a.cfg.FrontBaseURL, tld),
		url.QueryEscape(fromSettlement.TitleDefault),
		url.QueryEscape(toSettlement.TitleDefault),
	))
	if err != nil {
		return nil, NewStatusWithMessage(pb.EStatus_STATUS_INTERNAL_ERROR,
			fmt.Sprintf("%s: can not parse cfg.FrontBaseURL: %s", logMessage, err.Error()))
	}

	greenURL := frontURL.ResolveReference(
		&url.URL{
			RawQuery: wizSnippetGreenURLParams.Encode(),
		},
	).String()
	serpURL := frontURL.ResolveReference(
		&url.URL{
			RawQuery: wizSnippetSerpURLParams.Encode(),
		},
	).String()
	upcomingURL := frontURL.ResolveReference(
		&url.URL{
			RawQuery: wizSnippetUpcomingParams.Encode(),
		},
	).String()
	tomorrowURL := frontURL.ResolveReference(
		&url.URL{
			RawQuery: wizSnippetTomorrowParams.Encode(),
		},
	).String()

	siteLinks := []*ipb.TSiteLink{
		{
			Type: ipb.ESiteLinkType_SLT_UPCOMING,
			Url:  upcomingURL,
		},
		{
			Type: ipb.ESiteLinkType_SLT_TOMORROW,
			Url:  tomorrowURL,
		},
	}

	fromTitle := getSettlementTitle(fromSettlement)
	toTitle := getSettlementTitle(toSettlement)

	wizardResponse := &ipb.TWizardResponse{
		Favicon:      a.cfg.FrontFaviconURL,
		Rides:        rides,
		Additives:    additives,
		From:         from,
		FromName:     fromTitle,
		To:           to,
		ToName:       toTitle,
		MeanTime:     uint32(meanTime),
		GreenUrl:     greenURL,
		SerpUrl:      serpURL,
		SiteLinks:    siteLinks,
		MinPrice:     minPrices[0],
		SearchDate:   searchDate,
		OriginalDate: date,
		Title:        fmt.Sprintf("%s — %s: билеты на автобус", fromTitle, toTitle),
	}
	return wizardResponse, NewStatusWithMessage(pb.EStatus_STATUS_OK, "")
}

func (a *App) getPointKeyAndSettlement(geoID int32, pk *pb.TPointKey) (*pb.TPointKey, *rasp.TSettlement, *StatusWithMessage) {
	const logMessage = "App.getPointKeyAndSettlement"
	if pk == nil {
		settlement, ok := a.raspRepo.GetSettlementByGeoID(geoID)
		if !ok {
			return nil, nil, NewStatusWithMessage(pb.EStatus_STATUS_EXTERNAL_DICT_NOT_FOUND,
				fmt.Sprintf("%s: settlement with geoID=%d not found in RaspDict", logMessage, geoID))
		}
		newPK := &pb.TPointKey{
			Type: pb.EPointKeyType_POINT_KEY_TYPE_SETTLEMENT,
			Id:   uint32(settlement.Id),
		}
		return newPK, settlement, NewStatusWithMessage(pb.EStatus_STATUS_OK, "")
	}
	if pk.Type != pb.EPointKeyType_POINT_KEY_TYPE_SETTLEMENT {
		return nil, nil, NewStatusWithMessage(pb.EStatus_STATUS_BAD_REQUEST,
			fmt.Sprintf("%s: pointkey type %s is not allowed", logMessage, pk.Type))
	}
	settlement, ok := a.raspRepo.GetSettlement(int32(pk.Id))
	if !ok {
		return nil, nil, NewStatusWithMessage(pb.EStatus_STATUS_EXTERNAL_DICT_NOT_FOUND,
			fmt.Sprintf("%s: settlement with ID=%d not found in RaspDict", logMessage, settlement.Id))
	}
	return pk, settlement, NewStatusWithMessage(pb.EStatus_STATUS_OK, "")
}

func getSettlementTitle(settlement *rasp.TSettlement) string {
	if settlement.Title != nil && settlement.Title.Ru != nil && len(settlement.Title.Ru.Nominative) != 0 {
		return settlement.Title.Ru.Nominative
	}
	return settlement.TitleDefault
}
