package blacklist

import (
	"context"
	"fmt"
	"time"

	"a.yandex-team.ru/library/go/core/log/zap"
	"a.yandex-team.ru/toolbox/bunker"

	"a.yandex-team.ru/travel/buses/backend/internal/api/filters"
	"a.yandex-team.ru/travel/buses/backend/internal/common/logging"
	pb "a.yandex-team.ru/travel/buses/backend/proto"
)

type Config struct {
	BunkerPath string        `config:"blacklist-bunker-path"`
	BunkerNode string        `config:"blacklist-bunker-node"`
	Expiration time.Duration `config:"blacklist-expiration"`
}

var DefaultConfig = Config{
	BunkerPath: "sputnik-www/blacklist",
	BunkerNode: "production",
	Expiration: 10 * time.Minute,
}

var MockedConfig = Config{}

type (
	fetcher interface {
		run(context.Context, func(string))
	}
	bunkerFetcher struct {
		cfg           *Config
		logger        *zap.Logger
		bunkerOptions []bunker.Option
	}
	dummyFetcher struct {
		dataCallback func(data string)
	}
)

func (fetcher *bunkerFetcher) run(ctx context.Context, dataCallback func(string)) {
	var (
		node         = fetcher.cfg.BunkerNode
		bunkerClient = bunker.NewBunker(
			ctx,
			bunker.BunkerAPIURLProd,
			fetcher.cfg.BunkerPath,
			append([]bunker.Option{
				bunker.WithUpdateInterval(fetcher.cfg.Expiration),
				bunker.WithLogger(fetcher.logger),
			}, fetcher.bunkerOptions...)...,
		)
	)

	go func() {
		for {
			select {
			case bunkerNodes := <-bunkerClient.UpdatedCache():
				if data, ok := bunkerNodes["/"+node].(string); ok {
					dataCallback(data)
				} else {
					fetcher.logger.Errorf("no blacklist at node %s", node)
				}
			case <-ctx.Done():
				return
			}
		}
	}()
}

func (fetcher *dummyFetcher) run(_ context.Context, dataCallback func(data string)) {
	fetcher.dataCallback = dataCallback
}

type Blacklist struct {
	logger  *zap.Logger
	fetcher fetcher
	rules   rules
}

func NewBlacklist(cfg *Config, logger *zap.Logger) *Blacklist {
	var (
		blacklistLogger = logging.WithModuleContext(logger, "blacklist")
		fetcher         fetcher
	)

	if *cfg == MockedConfig {
		fetcher = &dummyFetcher{}
	} else {
		fetcher = &bunkerFetcher{cfg: cfg, logger: blacklistLogger}
	}
	return &Blacklist{logger: blacklistLogger, fetcher: fetcher}
}

func (blacklist *Blacklist) setRulesData(data string) error {
	var parsed, err = parseJSONRules(data)
	if err != nil {
		return fmt.Errorf("invalid rules json: %w", err)
	}

	blacklist.rules = newRules(parsed, blacklist.logger)
	return nil
}

func (blacklist *Blacklist) Apply(searchInfo *filters.SearchInfo, ride *pb.TRide) bool {
	for _, predicates := range blacklist.rules {
		var ruleMatches = true
		for _, predicate := range predicates {
			if !predicate(searchInfo, ride) {
				ruleMatches = false
				break
			}
		}
		if ruleMatches {
			return false
		}
	}
	return true
}

func (blacklist *Blacklist) RunFetcher(ctx context.Context) {
	blacklist.fetcher.run(ctx, func(data string) {
		if err := blacklist.setRulesData(data); err != nil {
			blacklist.logger.Errorf("Blacklist.RunFetcher: error getting rules: %s", err.Error())
		}
	})
}
