package blacklist

import (
	"fmt"
	"reflect"

	"a.yandex-team.ru/travel/buses/backend/internal/api/filters"
	"a.yandex-team.ru/travel/buses/backend/internal/common/dict"
	"a.yandex-team.ru/travel/buses/backend/internal/common/utils"
	pb "a.yandex-team.ru/travel/buses/backend/proto"
)

type comparableValue interface {
	isComparableValue()
}

type comparableString string

func (comparableString) isComparableValue() {}

type comparableInteger uint32

func (comparableInteger) isComparableValue() {}

type comparableIntegers map[uint32]bool

func (comparableIntegers) isComparableValue() {}

type comparablePointKey struct {
	Type pb.EPointKeyType
	ID   uint32
}

func (comparablePointKey) isComparableValue() {}

type converter interface {
	fromRuleValue(ruleValue interface{}) (comparableValue, error)
	fromSearchRide(searchInfo *filters.SearchInfo, ride *pb.TRide) comparableValue
}

type supplierConverter struct{}

func (supplierConverter) fromRuleValue(ruleValue interface{}) (comparableValue, error) {
	var supplierName, ok = ruleValue.(string)
	if !ok {
		return nil, fmt.Errorf("expected string, got %s", reflect.TypeOf(ruleValue))
	}
	var supplier, err = dict.GetSupplierByName(supplierName)
	if err != nil {
		return nil, err
	}
	return comparableInteger(supplier.ID), nil
}

func (supplierConverter) fromSearchRide(_ *filters.SearchInfo, ride *pb.TRide) comparableValue {
	return comparableInteger(ride.SupplierId)
}

type foundSuppliersConverter struct {
	supplierConverter
}

func (foundSuppliersConverter) fromSearchRide(searchInfo *filters.SearchInfo, _ *pb.TRide) comparableValue {
	return comparableIntegers(searchInfo.FoundSupplierIds)
}

type carrierNameConverter struct{}

func (carrierNameConverter) fromRuleValue(ruleValue interface{}) (comparableValue, error) {
	var carrierName, ok = ruleValue.(string)
	if !ok {
		return nil, fmt.Errorf("expected string, got %s", reflect.TypeOf(ruleValue))
	}
	return comparableString(carrierName), nil
}

func (carrierNameConverter) fromSearchRide(_ *filters.SearchInfo, ride *pb.TRide) comparableValue {
	return comparableString(ride.CarrierName)
}

type pointKeyConverter struct{}

func (converter pointKeyConverter) toComparable(pointKey *pb.TPointKey) comparablePointKey {
	return comparablePointKey{Type: pointKey.Type, ID: pointKey.Id}
}

func (converter pointKeyConverter) fromRuleValue(ruleValue interface{}) (comparableValue, error) {
	var stringPointKey, ok = ruleValue.(string)
	if !ok {
		return nil, fmt.Errorf("expected string, got %s", reflect.TypeOf(ruleValue))
	}
	var pointKey, err = utils.LoadPointKey(stringPointKey)
	if err != nil {
		return nil, fmt.Errorf("can't parse PointKey %s: %w", stringPointKey, err)
	}
	return converter.toComparable(pointKey), nil
}

type rideDeparturePointKeyConverter struct {
	pointKeyConverter
}

func (converter rideDeparturePointKeyConverter) fromSearchRide(_ *filters.SearchInfo, ride *pb.TRide) comparableValue {
	return converter.toComparable(ride.From)
}

type rideArrivalPointKeyConverter struct {
	pointKeyConverter
}

func (converter rideArrivalPointKeyConverter) fromSearchRide(_ *filters.SearchInfo, ride *pb.TRide) comparableValue {
	return converter.toComparable(ride.To)
}

type searchDeparturePointKeyConverter struct {
	pointKeyConverter
}

func (converter searchDeparturePointKeyConverter) fromSearchRide(searchInfo *filters.SearchInfo, _ *pb.TRide) comparableValue {
	return converter.toComparable(searchInfo.DeparturePK)
}

type searchArrivalPointKeyConverter struct {
	pointKeyConverter
}

func (converter searchArrivalPointKeyConverter) fromSearchRide(searchInfo *filters.SearchInfo, _ *pb.TRide) comparableValue {
	return converter.toComparable(searchInfo.ArrivalPK)
}
