package blacklist

import (
	"fmt"
	"strings"

	"a.yandex-team.ru/library/go/core/log/zap"
)

type rules [][]predicate

var fieldConverters = map[string]converter{
	"partner":        supplierConverter{},
	"carrier":        carrierNameConverter{},
	"from.id":        rideDeparturePointKeyConverter{},
	"from.serp_id":   searchDeparturePointKeyConverter{},
	"to.id":          rideArrivalPointKeyConverter{},
	"to.serp_id":     searchArrivalPointKeyConverter{},
	"found_partners": foundSuppliersConverter{},
}

var operatorPredicates = map[string]predicateFactory{
	"$eq":  predicateEquals,
	"$in":  makeContainedPredicate(true),
	"$nin": makeContainedPredicate(false),
}

func convertRuleToPredicates(r parsedRule) ([]predicate, error) {
	var result = make([]predicate, 0, len(r))

	for _, item := range r {
		var converter, ok = fieldConverters[strings.Join(item.path, ".")]
		if !ok {
			return nil, fmt.Errorf("unknown field %s", item.path)
		}

		if converter, ok := converter.(foundSuppliersConverter); ok {
			// TODO: implement operators on the found_partners field and remove this hack
			var predicate, err = predicateFoundSuppliers(item.value, converter)
			if err != nil {
				return nil, fmt.Errorf("cannot make predicate: %w", err)
			}
			result = append(result, predicate)
			continue
		}

		var (
			operatorName  = item.operator
			makePredicate predicateFactory
		)
		if operatorName == "" {
			makePredicate = predicateEquals
		} else {
			makePredicate, ok = operatorPredicates[operatorName]
			if !ok {
				return nil, fmt.Errorf("unknown operator %s", operatorName)
			}
		}

		var predicate, err = makePredicate(item.value, converter)
		if err != nil {
			return nil, fmt.Errorf("cannot make predicate: %w", err)
		}
		result = append(result, predicate)
	}

	return result, nil
}

func newRules(pp parsedRules, logger *zap.Logger) rules {
	var rr = make(rules, 0, len(pp))
	for _, p := range pp {
		var r, err = convertRuleToPredicates(p)
		if err != nil {
			logger.Errorf("blacklist.newRules: %s", err.Error())
			continue
		}
		if len(r) != 0 {
			rr = append(rr, r)
		}
	}
	return rr
}
