package cache

import (
	"fmt"
	"sync"
	"time"

	coremetrics "a.yandex-team.ru/library/go/core/metrics"
	"a.yandex-team.ru/travel/library/go/metrics"

	"a.yandex-team.ru/travel/buses/backend/internal/common/dict"
	ipb "a.yandex-team.ru/travel/buses/backend/internal/common/proto"
	pb "a.yandex-team.ru/travel/buses/backend/proto"
	wpb "a.yandex-team.ru/travel/buses/backend/proto/worker"
)

type SegmentKey struct {
	FromType pb.EPointKeyType
	FromID   uint32
	ToType   pb.EPointKeyType
	ToID     uint32
}

func makeSegmentKey(from *pb.TPointKey, to *pb.TPointKey) SegmentKey {
	return SegmentKey{
		FromType: from.Type,
		FromID:   from.Id,
		ToType:   to.Type,
		ToID:     to.Id,
	}
}

type SegmentsStorage struct {
	mutex        sync.RWMutex
	statuses     map[uint32]ipb.ECacheRecordStatus
	timestamps   map[uint32]time.Time
	segmentsSets map[uint32]map[SegmentKey]struct{}
	appMetrics   *metrics.AppMetrics
	gauges       map[uint32]coremetrics.Gauge
}

func NewSegmentsStorage(appMetrics *metrics.AppMetrics) *SegmentsStorage {
	return &SegmentsStorage{
		statuses:     map[uint32]ipb.ECacheRecordStatus{},
		timestamps:   map[uint32]time.Time{},
		segmentsSets: map[uint32]map[SegmentKey]struct{}{},
		appMetrics:   appMetrics,
		gauges:       map[uint32]coremetrics.Gauge{},
	}
}

func (ss *SegmentsStorage) updateMetrics(supplierID uint32) {
	if gauge, ok := ss.gauges[supplierID]; ok {
		gauge.Set(0)
	}
	supplier, err := dict.GetSupplier(supplierID)
	supplierName := fmt.Sprintf("Unknown:%d", supplierID)
	if err == nil {
		supplierName = supplier.Name
	}
	count := 0
	if s, ok := ss.segmentsSets[supplierID]; ok {
		count = len(s)
	}
	status, _ := ss.getStatus(supplierID)
	gauge := ss.appMetrics.GetOrCreateGauge(
		"segments_cache", map[string]string{
			"status":   status.String(),
			"supplier": supplierName,
		}, "count")
	gauge.Set(float64(count))
	ss.gauges[supplierID] = gauge
}

func (ss *SegmentsStorage) SetSegments(supplierID uint32, segments []*wpb.TSegment, timestamp time.Time) {
	ss.mutex.Lock()
	defer ss.mutex.Unlock()
	ss.segmentsSets[supplierID] = map[SegmentKey]struct{}{}
	for _, segment := range segments {
		ss.segmentsSets[supplierID][makeSegmentKey(segment.From, segment.To)] = struct{}{}
	}
	ss.statuses[supplierID] = ipb.ECacheRecordStatus_CACHE_RECORD_STATUS_OK
	ss.timestamps[supplierID] = timestamp
	ss.updateMetrics(supplierID)
}

func (ss *SegmentsStorage) SetStatus(supplierID uint32, status ipb.ECacheRecordStatus, timestamp time.Time) {
	ss.mutex.Lock()
	defer ss.mutex.Unlock()
	ss.statuses[supplierID] = status
	ss.timestamps[supplierID] = timestamp
	ss.updateMetrics(supplierID)
}

func (ss *SegmentsStorage) Has(supplierID uint32, from *pb.TPointKey, to *pb.TPointKey) bool {
	ss.mutex.RLock()
	defer ss.mutex.RUnlock()
	s, ok := ss.segmentsSets[supplierID]
	if !ok {
		return false
	}
	_, ok = s[makeSegmentKey(from, to)]
	return ok
}

func (ss *SegmentsStorage) getStatus(supplierID uint32) (ipb.ECacheRecordStatus, time.Time) {
	status, ok := ss.statuses[supplierID]
	if !ok {
		return ipb.ECacheRecordStatus_CACHE_RECORD_STATUS_MISSED, time.Time{}
	}
	ts := ss.timestamps[supplierID]
	return status, ts
}

func (ss *SegmentsStorage) GetStatus(supplierID uint32) (ipb.ECacheRecordStatus, time.Time) {
	ss.mutex.RLock()
	defer ss.mutex.RUnlock()
	return ss.getStatus(supplierID)
}
