package filters

import (
	"container/list"
	"context"
	"sync"
	"time"

	pb "a.yandex-team.ru/travel/buses/backend/proto"
)

type BannedRidesRule struct {
	bannedRidesStorage *BannedRidesStorage
}

func NewBannedRidesRule(ttl time.Duration) *BannedRidesRule {
	return &BannedRidesRule{
		bannedRidesStorage: NewBannedRideIDStorage(ttl),
	}
}

func (brf *BannedRidesRule) Apply(_ *SearchInfo, ride *pb.TRide) bool {
	return !brf.bannedRidesStorage.Exists(ride.Id)
}

func (brf *BannedRidesRule) Register(ride *pb.TRide) {
	brf.bannedRidesStorage.Register(ride.Id)
}

func (brf *BannedRidesRule) Run(ctx context.Context) {
	brf.bannedRidesStorage.Run(ctx)
}

func (brf *BannedRidesRule) Len() int {
	return brf.bannedRidesStorage.Len()
}

type BanListRecord struct {
	CreatedAt time.Time
	RideID    string
}

type BannedRidesStorage struct {
	mutex         sync.RWMutex
	ttl           time.Duration
	expiringQueue list.List
	storage       map[string]struct{}
}

func NewBannedRideIDStorage(ttl time.Duration) *BannedRidesStorage {
	return &BannedRidesStorage{
		ttl:     ttl,
		storage: make(map[string]struct{}),
	}
}

func (brs *BannedRidesStorage) Run(ctx context.Context) {
	go func() {
		for {
			var timer *time.Timer
			oldest := brs.expiringQueue.Front()
			if oldest == nil {
				timer = time.NewTimer(brs.ttl)
			} else {
				timer = time.NewTimer(brs.ttl - time.Since(oldest.Value.(*BanListRecord).CreatedAt))
			}

			select {
			case <-timer.C:
				oldest := brs.expiringQueue.Front()
				if oldest == nil {
					break
				}
				record := oldest.Value.(*BanListRecord)
				if brs.expired(record) {
					brs.mutex.Lock()
					brs.expiringQueue.Remove(oldest)
					delete(brs.storage, record.RideID)
					brs.mutex.Unlock()
				}
			case <-ctx.Done():
				return
			}
		}
	}()
}

func (brs *BannedRidesStorage) Len() int {
	return len(brs.storage)
}

func (brs *BannedRidesStorage) expired(record *BanListRecord) bool {
	recordAge := time.Since(record.CreatedAt)
	return recordAge > brs.ttl
}

func (brs *BannedRidesStorage) Register(rideID string) {
	brs.mutex.Lock()
	defer brs.mutex.Unlock()

	if _, ok := brs.storage[rideID]; ok {
		return
	}

	record := &BanListRecord{
		CreatedAt: time.Now(),
		RideID:    rideID,
	}
	brs.expiringQueue.PushBack(record)
	brs.storage[rideID] = struct{}{}
}

func (brs *BannedRidesStorage) Exists(rideID string) bool {
	brs.mutex.RLock()
	defer brs.mutex.RUnlock()

	_, ok := brs.storage[rideID]
	return ok
}
