package filters

import (
	"reflect"
	"time"

	"a.yandex-team.ru/library/go/core/log/zap"

	"a.yandex-team.ru/travel/buses/backend/internal/common/dict/rasp"
	"a.yandex-team.ru/travel/buses/backend/internal/common/utils"
	pb "a.yandex-team.ru/travel/buses/backend/proto"
)

type RideFilterRule interface {
	Apply(*SearchInfo, *pb.TRide) bool
}

type RidesFilter struct {
	rules []RideFilterRule
}

func NewRidesFilter(rules ...RideFilterRule) *RidesFilter {
	return &RidesFilter{
		rules: rules,
	}
}

func (rf *RidesFilter) Filter(departurePK *pb.TPointKey, arrivalPK *pb.TPointKey, rides []*pb.TRide, logger *zap.Logger) []*pb.TRide {
	searchInfo := newSearchInfo(departurePK, arrivalPK, rides)
	filteredRides := make([]*pb.TRide, len(rides))
	i := 0
Loop:
	for _, ride := range rides {
		for _, rule := range rf.rules {
			if !rule.Apply(searchInfo, ride) {
				logger.Debugf("RidesFilter.Filter: %s removed ride %s", reflect.TypeOf(rule), utils.RideStringer(ride))
				continue Loop
			}
		}
		filteredRides[i] = ride
		i++
	}
	if i < len(rides) {
		filteredRides = filteredRides[:i]
	}
	return filteredRides
}

type FreeSeatsRule struct{}

func (*FreeSeatsRule) Apply(_ *SearchInfo, ride *pb.TRide) bool {
	return ride.FreeSeats != 0
}

type RideStatusRule struct{}

func (*RideStatusRule) Apply(_ *SearchInfo, ride *pb.TRide) bool {
	return ride.Status == pb.ERideStatus_RIDE_STATUS_SALE
}

type DepartureInFutureRule struct {
	repo            *rasp.DictRepo
	defaultTimezone *time.Location
}

func NewDepartureInFutureRule(repo *rasp.DictRepo, defaultTimezone *time.Location) *DepartureInFutureRule {
	return &DepartureInFutureRule{
		repo:            repo,
		defaultTimezone: defaultTimezone,
	}
}

func (dif *DepartureInFutureRule) Apply(searchInfo *SearchInfo, ride *pb.TRide) bool {
	timezone, err := utils.GetPointKeyTimeZone(dif.repo, searchInfo.DeparturePK)
	if err != nil {
		timezone = dif.defaultTimezone
	}
	return time.Now().Before(utils.ConvertRideSecondsToTime(ride.DepartureTime, timezone))
}
