package grpcserver

import (
	"time"

	"a.yandex-team.ru/library/go/core/log/zap"
	coreMetrics "a.yandex-team.ru/library/go/core/metrics"
	grpcserver "a.yandex-team.ru/travel/library/go/grpcutil/server"
	"a.yandex-team.ru/travel/library/go/tvm"
	"google.golang.org/grpc"
	ghealth "google.golang.org/grpc/health"
	healthpb "google.golang.org/grpc/health/grpc_health_v1"

	"a.yandex-team.ru/travel/buses/backend/internal/api/app"
	commonGrpc "a.yandex-team.ru/travel/buses/backend/internal/common/grpc"
	apipb "a.yandex-team.ru/travel/buses/backend/proto/api"
)

type handler struct {
	apipb.UnimplementedApiServiceServer

	app    *app.App
	logger *zap.Logger
}

func New(app *app.App, cfg *commonGrpc.ServerConfig, logger *zap.Logger, metricsRegistry coreMetrics.Registry) *grpcserver.GrpcServer {
	healthServer := ghealth.NewServer()
	go func() {
		for range time.Tick(2 * time.Second) {
			healthServer.SetServingStatus("buses.backend.api", healthpb.HealthCheckResponse_SERVING)
		}
	}()

	var grpcInterceptors []grpc.UnaryServerInterceptor
	tvmHelper := tvm.NewDeployTvmHelper(logger, cfg.Tvm)
	if tvmInterceptor := tvmHelper.CheckServiceTicketGRPCInterceptor(); tvmInterceptor != nil {
		grpcInterceptors = append(grpcInterceptors, tvmInterceptor)
	}

	grpcServer := grpcserver.NewGrpcServerBuilder(grpcserver.GrpcConfig{Addr: cfg.Address}, logger).
		WithMetrics(metricsRegistry).
		WithHealthServer(healthServer).
		WithInterceptors(grpcInterceptors...).
		WithRegisterers(func(server *grpc.Server) {
			apipb.RegisterApiServiceServer(server, &handler{app: app, logger: logger})
		}).
		Build()

	return grpcServer
}
