package handler

import (
	pb "a.yandex-team.ru/travel/buses/backend/proto"
	"encoding/json"
	"fmt"
	"net/http"
	"net/url"
	"time"

	tpb "a.yandex-team.ru/travel/proto"

	"a.yandex-team.ru/travel/buses/backend/internal/api/app"
	"a.yandex-team.ru/travel/buses/backend/internal/common/connector"
	"a.yandex-team.ru/travel/buses/backend/internal/common/dict"
	"a.yandex-team.ru/travel/buses/backend/internal/common/utils"
)

func (h *Handler) Calendar(w http.ResponseWriter, r *http.Request) {
	params, _ := url.ParseQuery(r.URL.RawQuery)
	from, errFrom := utils.LoadPointKey(params.Get("from-id"))
	to, errTo := utils.LoadPointKey(params.Get("to-id"))
	minDate, errMinDate := loadMinDate(params.Get("min-date"))
	maxDate, errMaxDate := utils.LoadDate(params.Get("max-date"))
	if errFrom != nil || errTo != nil || errMinDate != nil || errMaxDate != nil {
		var msg string
		if errFrom != nil {
			msg = fmt.Sprintf("bad value for 'from-id': %s", errFrom.Error())
		} else if errTo != nil {
			msg = fmt.Sprintf("bad value for 'to-id': %s", errTo.Error())
		} else if errMinDate != nil {
			msg = fmt.Sprintf("bad value for 'min-date': %s", errMinDate.Error())
		} else if errMaxDate != nil {
			msg = fmt.Sprintf("bad value for 'max-date': %s", errMaxDate.Error())
		}
		w.WriteHeader(http.StatusBadRequest)
		_, _ = w.Write([]byte(msg))
		return
	}

	calendar, err := h.app.Calendar(from, to, minDate, maxDate, pb.ERequestSource_SRS_CALENDAR, r.Context())

	var response []byte
	responseStatus := http.StatusOK
	if err != nil {
		responseStatus = http.StatusBadRequest
		response, _ = json.Marshal(JSONErrorMapper{Error: err.Error()})
	} else {
		response, err = h.calendarToJSON(calendar)
		if err != nil {
			msg := fmt.Sprintf("marshaling error: %s", err.Error())
			h.logger.Error(msg)
			w.WriteHeader(http.StatusInternalServerError)
			_, _ = w.Write([]byte(msg))
			return
		}
	}

	w.Header().Set("Content-Type", "application/json")
	w.WriteHeader(responseStatus)
	_, _ = w.Write(response)
}

func loadMinDate(param string) (*tpb.TDate, error) {
	if param == "" {
		return utils.ConvertTimeToProtoDate(time.Now()), nil
	}
	return utils.LoadDate(param)
}

func (h *Handler) calendarToJSON(calendarItems []app.CalendarItem) ([]byte, error) {
	calendar := make(map[string]connector.JSONCalendarItemMapper, len(calendarItems))
	for _, item := range calendarItems {
		itemType := "empty"
		if item.RideCount != 0 {
			itemType = "valid"
		}

		minPrices := make(map[tpb.ECurrency]float64, len(item.MinPrices))
		for _, price := range item.MinPrices {
			minPrices[price.Currency] = float64(price.Amount) / 100
		}

		minPricesBySupplier := make(map[string]map[tpb.ECurrency]float64, len(item.MinPricesBySupplierIDs))
		for supplierID, supplierMinPrices := range item.MinPricesBySupplierIDs {
			minPrices := make(map[tpb.ECurrency]float64, len(supplierMinPrices))
			for _, price := range supplierMinPrices {
				minPrices[price.Currency] = float64(price.Amount) / 100
			}

			supplier, err := dict.GetSupplier(supplierID)
			if err != nil {
				return []byte("[]"), fmt.Errorf("dict error: %w", err)
			}
			minPricesBySupplier[supplier.Name] = minPrices
		}

		calendar[item.Date.Format(connector.DateLayout)] = connector.JSONCalendarItemMapper{
			Type:                itemType,
			RideCount:           item.RideCount,
			MinPrices:           minPrices,
			MinPricesBySupplier: minPricesBySupplier,
		}
	}
	calendarJSON, err := json.Marshal(calendar)
	if err != nil {
		return []byte("[]"), fmt.Errorf("JSON marshaling error: %w", err)
	}
	return calendarJSON, nil
}
