package handler

import (
	"encoding/json"
	"net/http"
	"net/url"
	"strconv"

	"a.yandex-team.ru/library/go/core/log/zap"

	"a.yandex-team.ru/travel/buses/backend/internal/api/app"
	"a.yandex-team.ru/travel/buses/backend/internal/common/logging"
	pb "a.yandex-team.ru/travel/buses/backend/proto"
)

type Handler struct {
	logger *zap.Logger
	app    *app.App
}

type JSONErrorMapper struct {
	Error string `json:"error"`
}

func New(app *app.App, logger *zap.Logger) *Handler {
	h := &Handler{
		logger: logging.WithModuleContext(logger, "handler"),
		app:    app,
	}
	return h
}

func (h *Handler) Ping(w http.ResponseWriter, _ *http.Request) {
	status := h.app.HealthStatus()
	w.WriteHeader(h.httpCodeForStatus(status))
	if !status.Ok() {
		w.Header().Set("Content-Type", "application/json")
		response, _ := json.Marshal(JSONErrorMapper{Error: status.String()})
		_, _ = w.Write(response)
		return
	}
	_, _ = w.Write([]byte("I'm fine, and you?"))
}

func (h *Handler) ReloadResources(w http.ResponseWriter, _ *http.Request) {
	status := h.app.ReloadResources()
	w.WriteHeader(h.httpCodeForStatus(status))
	if !status.Ok() {
		w.Header().Set("Content-Type", "application/json")
		response, _ := json.Marshal(JSONErrorMapper{Error: status.String()})
		_, _ = w.Write(response)
	}
}

func getAPIVersion(params url.Values) pb.EApiVersion {
	version, err := strconv.ParseInt(params.Get("v"), 10, 32)
	if err == nil {
		if _, ok := pb.EApiVersion_name[int32(version)]; ok {
			return pb.EApiVersion(version)
		}
	}
	return pb.EApiVersion_AV_DEFAULT
}

var httpCodes = map[pb.EStatus]int{
	pb.EStatus_STATUS_OK:                           http.StatusOK,
	pb.EStatus_STATUS_NOT_FOUND:                    http.StatusNotFound,
	pb.EStatus_STATUS_BAD_REQUEST:                  http.StatusBadRequest,
	pb.EStatus_STATUS_INTERNAL_ERROR:               http.StatusInternalServerError,
	pb.EStatus_STATUS_EXTERNAL_ERROR:               http.StatusBadGateway,
	pb.EStatus_STATUS_RIDE_BOOK_PARAMS_ERROR:       http.StatusBadGateway,
	pb.EStatus_STATUS_RIDE_SEARCH_FAILS:            http.StatusBadGateway,
	pb.EStatus_STATUS_RIDE_REFINEMENT_ERROR:        http.StatusBadGateway,
	pb.EStatus_STATUS_RIDE_DISAPPEARED:             http.StatusNotFound,
	pb.EStatus_STATUS_EXTERNAL_DICT_NOT_FOUND:      http.StatusConflict,
	pb.EStatus_STATUS_NOT_READY:                    http.StatusAccepted,
	pb.EStatus_STATUS_RIDE_SUBSTITUTION:            http.StatusOK,
	pb.EStatus_STATUS_RIDE_SUBSTITUTION_FROM_OTHER: http.StatusOK,
}

func (h *Handler) httpCodeForStatus(s *app.StatusWithMessage) int {
	const fnName = "Handler.httpCodeForStatus"
	if httpStatus, ok := httpCodes[s.Status]; ok {
		return httpStatus
	}
	h.logger.Errorf("%s: status %s is not supposed for http response", fnName, s.Status.String())
	return http.StatusInternalServerError
}
