package handler

import (
	"fmt"
	"net/url"

	"a.yandex-team.ru/travel/buses/backend/internal/common/connector"
	"a.yandex-team.ru/travel/buses/backend/internal/common/dict"
	"a.yandex-team.ru/travel/buses/backend/internal/common/utils"
	pb "a.yandex-team.ru/travel/buses/backend/proto"
)

var (
	registerTypeCodes = map[pb.ERegisterType]string{
		pb.ERegisterType_REGISTER_TYPE_UNKNOWN:      "",
		pb.ERegisterType_REGISTER_TYPE_COMPANY:      "ОГРН",
		pb.ERegisterType_REGISTER_TYPE_ENTREPRENEUR: "ОГРНИП",
	}
)

func registerTypeCodeByID(registerType pb.ERegisterType) string {
	registerTypeCode := registerTypeCodes[registerType]
	return registerTypeCode
}

func carrierToJSONCarrier(carrier *pb.TCarrier) *connector.Carrier {
	if carrier == nil {
		return nil
	}
	return &connector.Carrier{
		RegisterNumber: carrier.RegisterNumber,
		Name:           carrier.Name,
		Inn:            carrier.Inn,
		ActualAddress:  carrier.ActualAddress,
		LegalAddress:   carrier.LegalAddress,
		FirstName:      carrier.FirstName,
		MiddleName:     carrier.MiddleName,
		LastName:       carrier.LastName,
		RegisterType: connector.RegisterType{
			Code:        registerTypeCodeByID(carrier.RegisterType),
			Description: "",
		},
		LegalName: carrier.LegalName,
		Timetable: carrier.Timetable,
		ID:        carrier.ID,
	}
}

func benefitsToJSONBenefits(benefits []pb.EBenefitType) []connector.RideBenefit {
	jsonBenefits := make([]connector.RideBenefit, 0, len(benefits))
	for _, benefit := range benefits {
		if benefit != pb.EBenefitType_BENEFIT_TYPE_UNKNOWN {
			jsonBenefits = append(jsonBenefits, connector.RideBenefit(benefit))
		}
	}
	return jsonBenefits
}

func rideToJSONRide(ride *pb.TRide) (connector.JSONRide, error) {
	supplier, err := dict.GetSupplier(ride.SupplierId)
	if err != nil {
		return connector.JSONRide{}, fmt.Errorf("rideToJSONRide: dict error: %w", err)
	}
	fromID, _ := utils.DumpPointKey(ride.From)
	toID, _ := utils.DumpPointKey(ride.To)
	var yandexFee int64
	if ride.YandexFee != nil {
		yandexFee = ride.YandexFee.Amount
	}
	statusName := "unknown"
	if ride.Status == pb.ERideStatus_RIDE_STATUS_SALE {
		statusName = "sale"
	}
	convertedRide := connector.JSONRide{
		RideID:       ride.Id,
		Partner:      supplier.Name,
		CarrierID:    ride.CarrierCode,
		Carrier:      ride.CarrierName,
		CarrierModel: carrierToJSONCarrier(ride.Carrier),
		Arrival:      connector.RideTime(ride.ArrivalTime),
		Departure:    connector.RideTime(ride.DepartureTime),
		Duration:     int(ride.Duration),
		From: connector.RideEndpoint{
			ID:         fromID,
			Desc:       ride.FromDesc,
			SupplierID: ride.FromSupplierId,
		},
		To: connector.RideEndpoint{
			ID:         toID,
			Desc:       ride.ToDesc,
			SupplierID: ride.ToSupplierId,
		},
		Price:     float64(ride.Price.Amount) / 100,
		Fee:       float64(ride.Fee.Amount) / 100,
		YandexFee: float64(yandexFee) / 100,
		Currency:  "RUB", // TODO dict
		Status: connector.IDWithName{
			ID:   int(ride.Status),
			Name: statusName, //TODO dict or remove
		},
		FreeSeats:        int(ride.FreeSeats),
		Bus:              ride.Bus,
		CanPayOffline:    ride.CanPayOffline,
		BookOnly:         ride.BookOnly,
		Benefits:         benefitsToJSONBenefits(ride.Benefits),
		Name:             ride.RouteName,
		Number:           ride.RouteNumber,
		TicketLimit:      int(ride.TicketLimit),
		OnlineRefund:     ride.OnlineRefund,
		RefundConditions: ride.RefundConditions,
		BookFields:       ride.BookFields,
	}
	return convertedRide, nil
}

func ridesToJSONRides(rides []*pb.TRide) ([]connector.JSONRide, error) {
	const logMessage = "ridesToJSONRides"
	jsonRides := make([]connector.JSONRide, len(rides))
	i := 0
	for _, ride := range rides {
		jsonRide, err := rideToJSONRide(ride)
		if err != nil {
			return nil, fmt.Errorf("%s: %s", logMessage, err.Error())
		}
		jsonRides[i] = jsonRide
		i++
	}
	return jsonRides, nil
}

func RaspRides(rides []connector.JSONRide, from string, to string, date string) []connector.RaspJSONRide {
	raspRides := make([]connector.RaspJSONRide, len(rides))
	params := url.Values{}
	params.Set("utm_source", "rasp")
	params.Set("utm_medium", "buy_button")
	for i, ride := range rides {
		raspRide := connector.RaspJSONRide{
			JSONRide: ride,
			From:     ride.From.ID,
			To:       ride.To.ID,
		}
		raspRide.BookURL = utils.BookURL("", from, to, date, ride.RideID, params)
		if raspRide.From == "" || raspRide.From[0] != 's' {
			raspRide.From = from
		}
		if raspRide.To == "" || raspRide.To[0] != 's' {
			raspRide.To = to
		}
		raspRide.Number = "" // SPTNK-1825
		raspRides[i] = raspRide
	}
	return raspRides
}
