package handler

import (
	"encoding/json"
	"fmt"
	"net/http"
	"net/url"
	"strconv"

	tpb "a.yandex-team.ru/travel/proto"

	"a.yandex-team.ru/travel/buses/backend/internal/common/logging"
	"a.yandex-team.ru/travel/buses/backend/internal/common/utils"
	pb "a.yandex-team.ru/travel/buses/backend/proto"
)

type SearchParams struct {
	From *pb.TPointKey
	To   *pb.TPointKey
	Date *tpb.TDate
}

func (h *Handler) Search(w http.ResponseWriter, r *http.Request) {
	const logMessage = "Handler.Search"

	w.Header().Set("Content-Type", "application/json")

	params, _ := url.ParseQuery(r.URL.RawQuery)
	searchPrms, err := parseSearchParams(params)
	if err != nil {
		w.WriteHeader(http.StatusBadRequest)
		response, _ := json.Marshal(JSONErrorMapper{Error: err.Error()})
		_, _ = w.Write(response)
		return
	}
	deduplicate, errDeduplicate := strconv.ParseBool(params.Get("deduplicate"))
	if errDeduplicate != nil {
		deduplicate = true
	}
	rides, ready := h.app.Search(searchPrms.From, searchPrms.To, searchPrms.Date, deduplicate, pb.ERequestSource_SRS_SEARCH, r.Context())

	responseStatus := http.StatusOK
	if !ready {
		responseStatus = http.StatusAccepted
	}

	var response []byte
	apiVersion := getAPIVersion(params)
	switch apiVersion {
	case pb.EApiVersion_AV_V1:
		responseJSON, errJSON := ridesToJSONRides(rides)
		if errJSON == nil {
			response, err = json.Marshal(responseJSON)
		}
	case pb.EApiVersion_AV_V2:
		response, err = json.Marshal(rides)
	default:
		err = fmt.Errorf("not supported EApiVersion=%d", apiVersion)
	}

	if err != nil {
		logging.WithRequestContext(h.logger, r.Context()).Error(err.Error())
		w.WriteHeader(http.StatusInternalServerError)
		response, _ = json.Marshal(JSONErrorMapper{Error: err.Error()})
		_, _ = w.Write(response)
		return
	}

	w.WriteHeader(responseStatus)
	_, _ = w.Write(response)
}

func parseSearchParams(params url.Values) (*SearchParams, error) {
	from, errFrom := utils.LoadPointKey(params.Get("from-id"))
	if errFrom != nil {
		return &SearchParams{}, fmt.Errorf("bad value for 'from-id': %s", errFrom.Error())
	}
	to, errTo := utils.LoadPointKey(params.Get("to-id"))
	if errTo != nil {
		return &SearchParams{}, fmt.Errorf("bad value for 'to-id': %s", errTo.Error())
	}
	date, errDate := utils.LoadDate(params.Get("date"))
	if errDate != nil {
		return &SearchParams{}, fmt.Errorf("bad value for 'date': %s", errDate.Error())
	}
	return &SearchParams{
		From: from,
		To:   to,
		Date: date,
	}, nil
}
