package handler

import (
	"context"
	"encoding/json"
	"fmt"
	"net/http"
	"net/url"
	"strconv"

	"github.com/go-chi/chi/v5/middleware"

	"a.yandex-team.ru/travel/buses/backend/internal/common/connector"
	"a.yandex-team.ru/travel/buses/backend/internal/common/logging"
	ipb "a.yandex-team.ru/travel/buses/backend/internal/common/proto"
	"a.yandex-team.ru/travel/buses/backend/internal/common/utils"
	pb "a.yandex-team.ru/travel/buses/backend/proto"
	tpb "a.yandex-team.ru/travel/proto"
)

func (h *Handler) Wizard(w http.ResponseWriter, r *http.Request) {
	w.Header().Set("Content-Type", "application/json")

	params, _ := url.ParseQuery(r.URL.RawQuery)
	date, dateErr := utils.LoadDate(params.Get("date"))
	tld := params.Get("tld")
	reqID := params.Get("reqid")
	mainReqID := params.Get("main_reqid")
	fromGeoID, from, fromErr := geoOrPointKeyExtractor(params, "from-id", "from-rasp-id")
	toGeoID, to, toErr := geoOrPointKeyExtractor(params, "to-id", "to-rasp-id")

	if fromErr != nil || toErr != nil || tld == "" {
		var msg string
		if fromErr != nil {
			msg = fromErr.Error()
		} else if toErr != nil {
			msg = toErr.Error()
		} else if tld == "" {
			msg = "bad value for 'tld'"
		}
		w.WriteHeader(http.StatusBadRequest)
		response, _ := json.Marshal(JSONErrorMapper{Error: msg})
		_, _ = w.Write(response)
		return
	}
	if dateErr != nil {
		date = nil
	}

	requestCtx := r.Context()
	if reqID != "" {
		requestCtx = context.WithValue(requestCtx, middleware.RequestIDKey, reqID)
	}

	wizardResponse, status := h.app.Wizard(fromGeoID, from, toGeoID, to, date, tld, mainReqID, requestCtx)
	if !status.Ok() {
		logging.WithRequestContext(h.logger, requestCtx).Error(status.String())
		w.WriteHeader(h.httpCodeForStatus(status))
		response, _ := json.Marshal(JSONErrorMapper{Error: status.String()})
		_, _ = w.Write(response)
		return
	}

	response, err := h.wizardResponseToJSON(wizardResponse)
	if err != nil {
		msg := fmt.Sprintf("marshaling error: %s", err.Error())
		logging.WithRequestContext(h.logger, requestCtx).Error(msg)
		w.WriteHeader(http.StatusInternalServerError)
		response, _ := json.Marshal(JSONErrorMapper{Error: msg})
		_, _ = w.Write(response)
		return
	}

	w.WriteHeader(http.StatusOK)
	_, _ = w.Write(response)
}

type OriginalQuery struct {
	DepartureDate string `json:"departureDate"`
}

type SiteLink struct {
	URL  string `json:"url"`
	Type string `json:"type"`
}

type GreenURLStruct struct {
	Root string `json:"root"`
}

type WizardJSONResponse struct {
	Offers     []connector.JSONRide `json:"offers"`
	RacesCount int                  `json:"racesCount"`
	ToName     string               `json:"toName"`
	ToID       string               `json:"toId"`
	FromName   string               `json:"fromName"`
	FromID     string               `json:"fromId"`
	Title      string               `json:"title"`
	GreenURL   GreenURLStruct       `json:"greenUrl"`
	SerpURL    string               `json:"serpUrl"`
	SiteLinks  []SiteLink           `json:"sitelinks"`
	Favicon    string               `json:"favicon"`
	Currency   string               `json:"currency"`
	MinPrice   float64              `json:"minPrice"`
	MeanTime   uint32               `json:"meanTime"`
	Date       string               `json:"date"`
	Query      OriginalQuery        `json:"query"`
}

func geoOrPointKeyExtractor(params url.Values, geoIDArg string, pkArg string) (int32, *pb.TPointKey, error) {
	geoID, geoIDErr := strconv.ParseInt(params.Get(geoIDArg), 10, 32)
	if geoIDErr != nil {
		geoID = -1
	}
	pk, pkErr := utils.LoadPointKey(params.Get(pkArg))
	if pkErr != nil {
		pk = nil
	}
	if geoIDErr != nil && pkErr != nil {
		return int32(geoID), pk, fmt.Errorf(
			"geoOrPointKeyExtractor: one of %s, %s should be passed", geoIDArg, pkArg)
	}
	return int32(geoID), pk, nil
}

func (h *Handler) wizardResponseToJSON(response *ipb.TWizardResponse) ([]byte, error) {
	const logMessage = "Handler:wizardResponseToJSON"

	toID, _ := utils.DumpPointKey(response.To)
	fromID, _ := utils.DumpPointKey(response.From)
	siteLinks := make([]SiteLink, len(response.SiteLinks))
	i := 0
	for _, siteLink := range response.SiteLinks {
		var tp string
		if siteLink.Type == ipb.ESiteLinkType_SLT_UPCOMING {
			tp = "upcoming"
		} else if siteLink.Type == ipb.ESiteLinkType_SLT_TOMORROW {
			tp = "tomorrow"
		} else {
			return nil, fmt.Errorf("%s: unknown type %s", logMessage, siteLink.Type)
		}
		siteLinks[i] = SiteLink{
			URL:  siteLink.Url,
			Type: tp,
		}
		i++
	}
	if response.MinPrice.Currency != tpb.ECurrency_C_RUB {
		return nil, fmt.Errorf("%s: unsupported currency %s", logMessage, response.MinPrice.Currency)
	}

	jsonRides, err := ridesToJSONRides(response.Rides)
	if err != nil {
		return nil, fmt.Errorf("%s: %s", logMessage, err.Error())
	}
	if len(response.Rides) != len(response.Additives) {
		return nil, fmt.Errorf("%s: inconsistent wizard resonse", logMessage)
	}
	for i := range jsonRides {
		jsonRides[i].BookURLOrig = response.Additives[i].BookUrlOrig
		jsonRides[i].BookURL = response.Additives[i].BookUrl
	}

	originalDate := ""
	if response.OriginalDate != nil {
		originalDate = utils.DumpDate(response.OriginalDate)
	}

	wizardJSONResponse := WizardJSONResponse{
		Offers:     jsonRides,
		RacesCount: len(jsonRides),
		ToName:     response.ToName,
		ToID:       toID,
		FromName:   response.FromName,
		FromID:     fromID,
		Title:      response.Title,
		GreenURL: GreenURLStruct{
			Root: response.GreenUrl,
		},
		SerpURL:   response.SerpUrl,
		SiteLinks: siteLinks,
		Favicon:   response.Favicon,
		Currency:  "RUB",
		MinPrice:  float64(response.MinPrice.Amount) / 100,
		MeanTime:  response.MeanTime,
		Date:      utils.DumpDate(response.SearchDate),
		Query: OriginalQuery{
			DepartureDate: originalDate,
		},
	}

	jsonBytes, err := json.Marshal(wizardJSONResponse)
	if err != nil {
		return nil, fmt.Errorf("%s: JSON marshaling error: %w", logMessage, err)
	}
	return jsonBytes, nil
}
