package httpserver

import (
	"io"
	"net/http"
	"net/http/pprof"
	"strings"

	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/library/go/core/log/zap"
	coreMetrics "a.yandex-team.ru/library/go/core/metrics"
	"a.yandex-team.ru/library/go/httputil/middleware/httpmetrics"
	"a.yandex-team.ru/travel/library/go/tracing"
	"github.com/go-chi/chi/v5"
	chiMiddleware "github.com/go-chi/chi/v5/middleware"

	"a.yandex-team.ru/travel/buses/backend/internal/api/app"
	"a.yandex-team.ru/travel/buses/backend/internal/api/handler"
	"a.yandex-team.ru/travel/buses/backend/internal/api/middleware"
)

type Config struct {
	Address string `config:"server-address,required"`
}

type Server struct {
	cfg          *Config
	logger       *zap.Logger
	router       *chi.Mux
	tracerCloser io.Closer
}

var DefaultConfig = Config{
	Address: "127.0.0.1:9000",
}

func New(app *app.App, cfg *Config, logger *zap.Logger, metricsRegistry coreMetrics.Registry) (*Server, error) {
	server := &Server{
		cfg:          cfg,
		logger:       logger,
		router:       chi.NewRouter(),
		tracerCloser: tracing.InitializeDefaultTracer("buses.backend.api"),
	}

	tracingMiddleware := tracing.NewTracingMiddlewareBuilder().
		WithExtractor(tracing.NewHeaderTagExtractor("X-Request-Id", "request-id")).
		Build()

	apiMiddleware := middleware.New(logger)
	server.router.Use(chiMiddleware.RequestID)
	server.router.Use(tracingMiddleware.Handle)
	server.router.Use(apiMiddleware.Recover)
	server.router.Use(httpmetrics.New(metricsRegistry,
		httpmetrics.WithEndpointKey(func(req *http.Request) string {
			if strings.HasPrefix(req.URL.Path, "/points/") {
				return "/points/POINTKEY"
			}
			return req.URL.Path
		}),
		httpmetrics.WithSolomonRated(),
		httpmetrics.WithHTTPCodes(http.StatusOK, http.StatusAccepted, http.StatusNotFound,
			http.StatusInternalServerError, http.StatusBadGateway, http.StatusConflict, http.StatusBadRequest)))

	h := handler.New(app, logger)
	server.router.Get("/api/calendar", h.Calendar)
	server.router.Get("/api/ping", h.Ping)
	server.router.Get("/api/popular", h.Popular)
	server.router.Get("/api/reload_resources", h.ReloadResources)
	server.router.Get("/api/ride", h.Ride)
	server.router.Get("/api/search", h.Search)
	server.router.Get("/api/search/rasp", h.RaspSearch)

	server.router.Get("/api/wizard", h.Wizard)
	server.router.Get("/wizard/", h.Wizard) // legacy url

	server.router.Get("/points/{pointkey}", h.PointByPointkey)
	server.router.Get("/points", h.Points)

	server.router.Get("/debug/pprof/*", pprof.Index)

	return server, nil
}

func (server *Server) ListenAndServe() error {
	address := server.cfg.Address
	server.logger.Info("Start listening", log.String("address", address))
	err := http.ListenAndServe(address, server.router)
	if err != nil {
		server.logger.Error("Server stopped listening with error", log.Error(err))
	}
	return err
}

func (server *Server) Close() error {
	return server.tracerCloser.Close()
}
