package middleware

import (
	"fmt"
	"net/http"
	"time"

	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/library/go/core/log/ctxlog"
	"a.yandex-team.ru/library/go/core/log/zap"

	"a.yandex-team.ru/travel/buses/backend/internal/common/logging"
)

type Middleware struct {
	logger *zap.Logger
}

func New(logger *zap.Logger) *Middleware {
	h := &Middleware{
		logger: logging.WithModuleContext(logger, "middleware"),
	}
	return h
}

func (m *Middleware) Logger(next http.Handler) http.Handler {
	fn := func(w http.ResponseWriter, r *http.Request) {
		t := time.Now()
		defer func() {
			msg := fmt.Sprintf("Access: %s", r.RequestURI)
			m.logger.Info(msg,
				log.Time("time", t),
				log.Duration("duration", time.Since(t)),
				log.String("uri", r.RequestURI),
				log.String("remote", r.RemoteAddr),
			)
		}()
		next.ServeHTTP(w, r)
	}
	return http.HandlerFunc(fn)
}

func (m *Middleware) Recover(next http.Handler) http.Handler {
	return http.HandlerFunc(func(w http.ResponseWriter, req *http.Request) {
		defer func() {
			if r := recover(); r != nil {
				w.WriteHeader(http.StatusInternalServerError)
				errorMessage := fmt.Sprintf("An error is caught: %+v", r)
				ctxlog.Error(req.Context(), m.logger, errorMessage)
			}
		}()
		next.ServeHTTP(w, req)
	})
}
