package offerstorage

import (
	"context"
	"encoding/json"
	"errors"
	"fmt"
	"path"
	"time"

	"a.yandex-team.ru/kikimr/public/sdk/go/ydb"
	"a.yandex-team.ru/kikimr/public/sdk/go/ydb/table"
	"github.com/gofrs/uuid"
	"github.com/opentracing/opentracing-go"

	apipb "a.yandex-team.ru/travel/buses/backend/proto/api"
)

const (
	OffersTableName = "buses_offers"
)

func (s *Storage) CreateOffersTable(ctx context.Context) error {
	err := s.doRetrying(ctx, func(ctx context.Context, session *table.Session) error {
		return session.CreateTable(ctx, path.Join(s.Config.Database, OffersTableName),
			table.WithColumn("id", ydb.Optional(ydb.TypeUTF8)),
			table.WithColumn("created", ydb.Optional(ydb.TypeTimestamp)),
			table.WithColumn("data", ydb.Optional(ydb.TypeJSON)),
			table.WithPrimaryKeyColumn("id"),
		)
	})
	if err != nil {
		return fmt.Errorf("ydb create table error: %w", err)
	}
	return nil
}

var insertQuery = fmt.Sprintf(`
DECLARE $id AS Utf8;
DECLARE $created AS Timestamp;
DECLARE $data AS Json;

INSERT INTO %s (id, created, data)
VALUES ($id, $created, $data);
`, OffersTableName)

func (s *Storage) SaveOffer(ctx context.Context, offer *apipb.TOffer) (offerID uuid.UUID, err error) {
	if !s.Config.Enabled {
		return offerID, err
	}
	span, ctx := opentracing.StartSpanFromContext(ctx, "storage.SaveOffer")
	defer span.Finish()

	offerID = uuid.Must(uuid.NewV4())
	offer.Id = offerID.String()
	data, err := json.Marshal(offer)
	if err != nil {
		return offerID, fmt.Errorf("offer marshal error: %w", err)
	}

	ydbNow := ydb.Time(time.Now())
	err = s.doRetrying(ctx, func(ctx context.Context, session *table.Session) (err error) {
		stmt, err := session.Prepare(ctx, insertQuery)
		if err != nil {
			return err
		}
		_, _, err = stmt.Execute(ctx, s.WriteTx, table.NewQueryParameters(
			table.ValueParam("$id", ydb.UTF8Value(offerID.String())),
			table.ValueParam("$created", ydb.TimestampValue(ydbNow.Timestamp())),
			table.ValueParam("$data", ydb.JSONValue(string(data))),
		))
		return err
	})
	if err != nil {
		return offerID, fmt.Errorf("ydb insert error: %w", err)
	}
	return offerID, nil
}

var selectQuery = fmt.Sprintf(`
DECLARE $id AS Utf8;

SELECT data
FROM %s
WHERE id = $id;
`, OffersTableName)

func (s *Storage) GetOffer(ctx context.Context, id uuid.UUID) (*apipb.TOffer, error) {
	if !s.Config.Enabled {
		return nil, errors.New("offer storage is disabled")
	}
	span, ctx := opentracing.StartSpanFromContext(ctx, "storage.GetOffer")
	defer span.Finish()
	var res *table.Result
	err := s.doRetrying(ctx, func(ctx context.Context, session *table.Session) (err error) {
		stmt, err := session.Prepare(ctx, selectQuery)
		if err != nil {
			return err
		}
		_, res, err = stmt.Execute(ctx, s.ReadTx, table.NewQueryParameters(
			table.ValueParam("$id", ydb.UTF8Value(id.String())),
		))
		return err
	})
	if err != nil {
		return nil, fmt.Errorf("ydb select error: %w", err)
	}
	for res.NextSet() {
		for res.NextRow() {
			res.SeekItem("data")
			data := res.OJSON()
			offer := new(apipb.TOffer)
			err = json.Unmarshal([]byte(data), offer)
			if err != nil {
				return nil, fmt.Errorf("offer unmarshal error: %w", err)
			} else {
				return offer, nil
			}
		}
	}
	return nil, fmt.Errorf("not found offer: %v", id)
}
