package offerstorage

import (
	"context"
	"fmt"
	"path"
	"time"

	"a.yandex-team.ru/kikimr/public/sdk/go/ydb"
	"a.yandex-team.ru/kikimr/public/sdk/go/ydb/table"
	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/travel/library/go/vault"
)

type Config struct {
	Endpoint  string        `config:"offer-storage-endpoint,required" yaml:"endpoint"`
	Database  string        `config:"offer-storage-database,required" yaml:"database"`
	Timeout   time.Duration `config:"offer-storage-timeout,required" yaml:"timeout"`
	SecretID  string        `config:"offer-storage-secret-id,required" yaml:"secret-id"`
	SecretKey string        `config:"offer-storage-secret-key,required" yaml:"secret-key"`
	Enabled   bool          `config:"offer-storage-enabled" yaml:"enabled"`
}

var DefaultConfig = Config{
	Timeout:   time.Second,
	SecretID:  "sec-01d956rm0wknnrbhhjsyx0rh56",
	SecretKey: "sputnik-yql-token",
	Enabled:   true,
}

type Storage struct {
	Logger      log.Logger
	Config      *Config
	SessionPool *table.SessionPool
	Retryer     *table.Retryer
	ReadTx      *table.TransactionControl
	WriteTx     *table.TransactionControl
}

func NewStorage(ctx context.Context, config *Config, logger log.Logger) (*Storage, error) {
	if !config.Enabled {
		return &Storage{
			Logger: logger,
			Config: config,
		}, nil
	}
	vaultResolver := vault.NewYavSecretsResolver()
	ydbToken, err := vaultResolver.GetSecretValue(config.SecretID, config.SecretKey)
	if err != nil {
		return nil, fmt.Errorf("get ydb token error: %w", err)
	}
	ydbConf := &ydb.DriverConfig{
		Credentials: ydb.AuthTokenCredentials{
			AuthToken: ydbToken,
		},
		Database:       config.Database,
		RequestTimeout: config.Timeout,
		StreamTimeout:  config.Timeout,
	}
	dialer := &ydb.Dialer{
		DriverConfig: ydbConf,
		Timeout:      config.Timeout,
	}
	driver, err := dialer.Dial(ctx, config.Endpoint)
	if err != nil {
		return nil, fmt.Errorf("dial error: %w", err)
	}
	tableClient := table.Client{
		Driver: driver,
	}
	sp := table.SessionPool{
		IdleThreshold:      time.Minute,
		Builder:            &tableClient,
		KeepAliveBatchSize: -1,
		SizeLimit:          3,
	}
	readTx := table.TxControl(
		table.BeginTx(
			table.WithOnlineReadOnly(),
		),
		table.CommitTx(),
	)
	writeTx := table.TxControl(
		table.BeginTx(
			table.WithSerializableReadWrite(),
		),
		table.CommitTx(),
	)
	retryer := &table.Retryer{
		MaxRetries:      3,
		Backoff:         ydb.BackoffFunc(func(n int) <-chan time.Time { return time.After(0) }),
		SessionProvider: &sp,
		RetryChecker:    ydb.RetryChecker{RetryNotFound: true},
	}

	storage := Storage{
		Logger:      logger,
		SessionPool: &sp,
		Config:      config,
		ReadTx:      readTx,
		WriteTx:     writeTx,
		Retryer:     retryer,
	}
	go func() {
		<-ctx.Done()
		_ = storage.Close(ctx)
	}()
	err = storage.describeTable(ctx, OffersTableName)
	if err != nil {
		if ydb.IsOpError(err, ydb.StatusNotFound) || ydb.IsOpError(err, ydb.StatusSchemeError) {
			err = storage.CreateOffersTable(ctx)
			if err != nil {
				return nil, err
			}
		}
	}
	return &storage, nil
}

func (s *Storage) Close(ctx context.Context) error {
	if s.SessionPool != nil {
		return s.SessionPool.Close(ctx)
	}
	return nil
}

func (s *Storage) describeTable(ctx context.Context, tableName string) error {
	err := s.doRetrying(ctx, func(ctx context.Context, session *table.Session) error {
		_, err := session.DescribeTable(ctx, path.Join(s.Config.Database, tableName))
		if err != nil {
			return err
		}
		return nil
	})
	return err
}

func (s *Storage) doRetrying(ctx context.Context, operation table.OperationFunc) error {
	return s.Retryer.Do(ctx, operation)
}
