package warmer

import (
	"context"
	"sync"
	"time"

	"a.yandex-team.ru/library/go/core/log/zap"
	coreMetrics "a.yandex-team.ru/library/go/core/metrics"
	"a.yandex-team.ru/travel/library/go/metrics"
	"a.yandex-team.ru/travel/library/go/resourcestorage"
	tpb "a.yandex-team.ru/travel/proto"

	"a.yandex-team.ru/travel/buses/backend/internal/api/cache"
	"a.yandex-team.ru/travel/buses/backend/internal/common/dict"
	"a.yandex-team.ru/travel/buses/backend/internal/common/dict/rasp"
	"a.yandex-team.ru/travel/buses/backend/internal/common/logging"
	"a.yandex-team.ru/travel/buses/backend/internal/common/popular"
	ipb "a.yandex-team.ru/travel/buses/backend/internal/common/proto"
	"a.yandex-team.ru/travel/buses/backend/internal/common/utils"
	pb "a.yandex-team.ru/travel/buses/backend/proto"
	wpb "a.yandex-team.ru/travel/buses/backend/proto/worker"
)

const (
	moduleName                        = "warmer"
	wizardPopularDirectionsResource   = "warmer_wizard_popular_directions"
	searchPopularDirectionsResource   = "warmer_search_popular_directions"
	popularDirectionsKeepLastVersions = 10
)

type Config struct {
	WizardDirectionsTop           int           `config:"warmer-wizarddirectionstop,required"`
	SearchDirectionsTop           int           `config:"warmer-searchdirectionstop,required"`
	CalendarDepth                 int           `config:"warmer-calendardepth,required"`
	CalendarK                     int           `config:"warmer-calendark,required"`
	NextStepWait                  time.Duration `config:"warmer-nextstepwait,required"`
	MaxTaskQueueDuration          time.Duration `config:"warmer-maxtaskqueueduration,required"`
	DumpPeriod                    time.Duration `config:"warmer-dumpperiod,required"`
	PopularDirectionRebuildPeriod time.Duration `config:"warmer-populardirectionrebuildperiod,required"`
	DefaultTimeZone               string        `config:"warmer-defaulttimezone,required"`
}

var DefaultConfig = Config{
	WizardDirectionsTop:           40000,
	SearchDirectionsTop:           15000,
	CalendarDepth:                 14,
	CalendarK:                     50,
	NextStepWait:                  15 * time.Minute,
	MaxTaskQueueDuration:          10 * time.Minute,
	DumpPeriod:                    1 * time.Hour,
	PopularDirectionRebuildPeriod: 15 * time.Minute,
	DefaultTimeZone:               "Europe/Moscow",
}

type Warmer struct {
	cfg                           Config
	logger                        *zap.Logger
	searchCache                   *cache.SearchRecordStorage
	segments                      *cache.SegmentsStorage
	wizardPopularDirections       *popular.Directions
	wizardPopularDirectionsDumper *resourcestorage.Dumper
	searchPopularDirections       *popular.Directions
	searchPopularDirectionsDumper *resourcestorage.Dumper
	workerServiceClient           wpb.WorkerServiceClient
	mutex                         sync.Mutex
	appMetrics                    *metrics.AppMetrics
	raspRepo                      *rasp.DictRepo
	defaultTimeLocation           *time.Location
	scheduled                     uint32
	supplierIDs                   []uint32
}

func NewWarmer(cfg Config, searchCache *cache.SearchRecordStorage, segments *cache.SegmentsStorage,
	workerServiceClient wpb.WorkerServiceClient, s3StorageCfg resourcestorage.S3StorageConfig,
	s3StorageAccessKey string, s3StorageSecret string,
	raspRepo *rasp.DictRepo, appMetrics *metrics.AppMetrics, supplierIDs []uint32, logger *zap.Logger) *Warmer {
	const logMessage = "NewWarmer"

	moduleLogger := logging.WithModuleContext(logger, moduleName)

	defaultTimeLocation, err := time.LoadLocation(cfg.DefaultTimeZone)
	if err != nil {
		moduleLogger.Errorf("%s: can not load timezone %s. Continue for UTC", logMessage, cfg.DefaultTimeZone)
		defaultTimeLocation = time.UTC
	}

	wizardPopularDirectionsCfg := popular.DefaultConfig
	wizardPopularDirectionsCfg.TopSize = cfg.WizardDirectionsTop
	wizardPopularDirectionsCfg.TopSizeFrom = 0
	wizardPopularDirectionsCfg.RebuildPeriod = cfg.PopularDirectionRebuildPeriod
	wizardPopularDirections := popular.NewDirections(wizardPopularDirectionsCfg, moduleLogger)

	searchPopularDirectionsCfg := wizardPopularDirectionsCfg
	searchPopularDirectionsCfg.TopSize = cfg.SearchDirectionsTop
	searchPopularDirections := popular.NewDirections(searchPopularDirectionsCfg, moduleLogger)

	s3StorageReader := resourcestorage.NewS3StorageReader(s3StorageCfg, s3StorageAccessKey, s3StorageSecret)

	wizardPopularDirectionsLoader := resourcestorage.NewLoader(&ipb.TPopularDirection{},
		wizardPopularDirectionsResource, s3StorageReader, moduleLogger)
	n, err := wizardPopularDirectionsLoader.Load(wizardPopularDirections)
	if err != nil {
		moduleLogger.Errorf("%s: can not load snapshot for %s: %s",
			logMessage, wizardPopularDirectionsResource, err.Error())
	} else {
		moduleLogger.Infof("%s: loaded %d records for %s", logMessage, n, wizardPopularDirectionsResource)
	}

	searchPopularDirectionsLoader := resourcestorage.NewLoader(&ipb.TPopularDirection{},
		searchPopularDirectionsResource, s3StorageReader, moduleLogger)
	n, err = searchPopularDirectionsLoader.Load(searchPopularDirections)
	if err != nil {
		moduleLogger.Errorf("%s: can not load snapshot for %s: %s",
			logMessage, searchPopularDirectionsResource, err.Error())
	} else {
		moduleLogger.Infof("%s: loaded %d records for %s", logMessage, n, wizardPopularDirectionsResource)
	}

	return &Warmer{
		cfg:                     cfg,
		logger:                  moduleLogger,
		searchCache:             searchCache,
		segments:                segments,
		wizardPopularDirections: wizardPopularDirections,
		wizardPopularDirectionsDumper: resourcestorage.NewDumper(wizardPopularDirections, wizardPopularDirectionsResource,
			resourcestorage.NewS3StorageWriter(s3StorageCfg, s3StorageAccessKey, s3StorageSecret),
			popularDirectionsKeepLastVersions, moduleLogger),
		searchPopularDirections: wizardPopularDirections,
		searchPopularDirectionsDumper: resourcestorage.NewDumper(searchPopularDirections, searchPopularDirectionsResource,
			resourcestorage.NewS3StorageWriter(s3StorageCfg, s3StorageAccessKey, s3StorageSecret),
			popularDirectionsKeepLastVersions, moduleLogger),
		workerServiceClient: workerServiceClient,
		mutex:               sync.Mutex{},
		appMetrics:          appMetrics,
		raspRepo:            raspRepo,
		defaultTimeLocation: defaultTimeLocation,
		scheduled:           0,
		supplierIDs:         supplierIDs,
	}
}

func (w *Warmer) RegisterQuery(from *pb.TPointKey, to *pb.TPointKey, source pb.ERequestSource) {
	if source == pb.ERequestSource_SRS_WIZARD {
		w.wizardPopularDirections.Register(from, to)
	} else if source == pb.ERequestSource_SRS_SEARCH {
		w.searchPopularDirections.Register(from, to)
	}
}

func (w *Warmer) Run(ctx context.Context) {
	w.wizardPopularDirectionsDumper.RunPeriodic(w.cfg.DumpPeriod, ctx)
	w.wizardPopularDirections.Run(ctx)
	w.searchPopularDirectionsDumper.RunPeriodic(w.cfg.DumpPeriod, ctx)
	w.searchPopularDirections.Run(ctx)

	supplierIDs := w.supplierIDs
	if len(supplierIDs) == 0 {
		supplierIDs = dict.GetSuppliersList()
	}
	for _, supplierID := range supplierIDs {
		supplier, _ := dict.GetSupplier(supplierID)
		go w.runSupplierLoop(supplier, ctx)
	}
}

func (w *Warmer) runSupplierLoop(supplier dict.Supplier, ctx context.Context) {
	metricsTags := map[string]string{"supplier": supplier.Name}
	startedGauge := w.appMetrics.GetOrCreateGauge("warmer", metricsTags, "started")

	for {
		startedGauge.Set(1)
		w.mutex.Lock()
		for !w.pushAllWizardTasks(supplier) || !w.pushAllCalendarTasks(supplier) {
			w.mutex.Unlock()
			timer := time.NewTimer(w.cfg.MaxTaskQueueDuration)
			select {
			case <-timer.C:
			case <-ctx.Done():
				return
			}
			w.mutex.Lock()
		}
		w.mutex.Unlock()
		startedGauge.Set(0)

		timer := time.NewTimer(w.cfg.NextStepWait)
		select {
		case <-timer.C:
		case <-ctx.Done():
			return
		}
	}
}

func (w *Warmer) pushAllWizardTasks(supplier dict.Supplier) bool {
	const logMessage = "Warmer.pushAllWizardTasks"

	if status, _ := w.segments.GetStatus(supplier.ID); status == ipb.ECacheRecordStatus_CACHE_RECORD_STATUS_MISSED {
		w.logger.Infof("%s: no segments for %s. Skipped", logMessage, supplier.Name)
		return true
	}

	metricsTags := map[string]string{"supplier": supplier.Name, "strategy": "WIZARD"}
	checkedGauge := w.appMetrics.GetOrCreateGauge("warmer", metricsTags, "checked")
	scheduledGauge := w.appMetrics.GetOrCreateGauge("warmer", metricsTags, "scheduled")

	today := time.Now()
	tomorrow := today.AddDate(0, 0, 1)
	for _, direction := range w.wizardPopularDirections.GetDirections() {
		if !w.segments.Has(supplier.ID, direction.From, direction.To) {
			continue
		}
		timeLocationFrom, err := utils.GetPointKeyTimeZone(w.raspRepo, direction.From)
		if err != nil {
			timeLocationFrom = w.defaultTimeLocation
		}
		for _, tm := range []time.Time{today, tomorrow} {
			date := utils.ConvertTimeToProtoDate(tm.In(timeLocationFrom))
			if !w.checkCacheAndSchedule(supplier, direction, date, checkedGauge, scheduledGauge) {
				return false
			}
		}
	}
	return true
}

func (w *Warmer) genWalkOrder(width, height int) [][]int {
	order := make([][]int, width*height)
	i := 0
	for diagonalI := 0; diagonalI < height/w.cfg.CalendarK+width; diagonalI++ {
		for x := 0; x <= diagonalI && x < width; x++ {
			for y := (diagonalI - x) * w.cfg.CalendarK; (y < (diagonalI-x+1)*w.cfg.CalendarK) && (y < height); y++ {
				order[i] = []int{x, y}
				i++
			}
		}
	}
	return order
}

func (w *Warmer) pushAllCalendarTasks(supplier dict.Supplier) bool {
	const logMessage = "Warmer.pushAllCalendarTasks"

	if !supplier.CalendarWarmer {
		return true
	}

	if status, _ := w.segments.GetStatus(supplier.ID); status == ipb.ECacheRecordStatus_CACHE_RECORD_STATUS_MISSED {
		w.logger.Infof("%s: no segments for %s. Skipped", logMessage, supplier.Name)
		return true
	}

	metricsTags := map[string]string{"supplier": supplier.Name, "strategy": "CALENDAR"}
	checkedGauge := w.appMetrics.GetOrCreateGauge("warmer", metricsTags, "checked")
	scheduledGauge := w.appMetrics.GetOrCreateGauge("warmer", metricsTags, "scheduled")

	now := time.Now().In(w.defaultTimeLocation)
	dates := make([]*tpb.TDate, w.cfg.CalendarDepth)
	for i := 0; i < w.cfg.CalendarDepth; i++ {
		dates[i] = utils.ConvertTimeToProtoDate(now.AddDate(0, 0, i))
	}
	directions := w.wizardPopularDirections.GetDirections()
	i := 0
	for _, direction := range directions {
		if w.segments.Has(supplier.ID, direction.From, direction.To) {
			directions[i] = direction
			i++
		}
	}
	directions = directions[:i]

	for _, xY := range w.genWalkOrder(w.cfg.CalendarDepth, len(directions)) {
		date := dates[xY[0]]
		direction := directions[xY[1]]
		if !w.checkCacheAndSchedule(supplier, direction, date, checkedGauge, scheduledGauge) {
			return false
		}
	}
	return true
}

func (w *Warmer) checkCacheAndSchedule(supplier dict.Supplier, direction *ipb.TPopularDirection, date *tpb.TDate,
	checkedGauge, scheduledGauge coreMetrics.Gauge) bool {
	const logMessage = "Warmer.checkCacheAndSchedule"

	checkedGauge.Add(1)
	record, ok := w.searchCache.Get(cache.NewSearchKey(supplier.ID, direction.From, direction.To, date))
	if ok {
		if record.Status == ipb.ECacheRecordStatus_CACHE_RECORD_STATUS_OK ||
			record.Status == ipb.ECacheRecordStatus_CACHE_RECORD_STATUS_REQUESTED ||
			record.Status == ipb.ECacheRecordStatus_CACHE_RECORD_STATUS_FAILED {
			return true
		}
	}

	searchResponse, err := w.workerServiceClient.Search(context.Background(), &wpb.TSearchRequest{
		Header: &wpb.TRequestHeader{
			Priority: wpb.ERequestPriority_REQUEST_PRIORITY_LOW,
		},
		SupplierId: supplier.ID,
		From:       direction.From,
		To:         direction.To,
		Date:       date,
	})
	if err != nil {
		w.logger.Errorf("%s: %s", logMessage, err.Error())
		return false
	}
	if searchResponse.Header.Code != tpb.EErrorCode_EC_OK {
		if searchResponse.Header.Error != nil {
			w.logger.Errorf("%s: %s", logMessage, searchResponse.Header.Error.Message)
		} else {
			w.logger.Errorf("%s: %s", logMessage, searchResponse.Header.Code.String())
		}
		return false
	}

	scheduledGauge.Add(1)
	w.scheduled++
	return searchResponse.QueuePosition < uint32(supplier.SearchRPS*w.cfg.MaxTaskQueueDuration.Seconds())
}
