package connector

import (
	"bytes"
	"encoding/json"
	"fmt"
	"io"
	"io/ioutil"
	"net/http"
	"time"

	pb "a.yandex-team.ru/travel/buses/backend/proto"

	tpb "a.yandex-team.ru/travel/proto"

	wpb "a.yandex-team.ru/travel/buses/backend/proto/worker"
)

const (
	TimeLayout = "2006-01-02T15:04:05"
	DateLayout = "2006-01-02"

	nullJSONValue = "null"

	priceCurrency  = tpb.ECurrency_C_RUB
	pricePrecision = 2
)

func readJSON(reader io.Reader, target interface{}) error {
	var data, err = ioutil.ReadAll(reader)
	if err != nil {
		return err
	}
	return json.Unmarshal(data, target)
}

type IDWithName struct {
	ID   int    `json:"id"`
	Name string `json:"name"`
}

func (in *IDWithName) IsEmpty() bool {
	return (*in) == (IDWithName{})
}

type RideEndpoint struct {
	ID         string `json:"id"`
	Desc       string `json:"desc"`
	SupplierID string `json:"supplier_id"`
}

type RideTime int64

func (rt RideTime) MarshalJSON() ([]byte, error) {
	if rt == 0 {
		return []byte(nullJSONValue), nil
	}
	return json.Marshal(time.Unix(int64(rt), 0).In(time.UTC).Format(TimeLayout))
}

func (rt *RideTime) UnmarshalJSON(input []byte) error {
	if bytes.Equal(input, []byte(nullJSONValue)) {
		return nil
	}

	var timeStr string
	if err := json.Unmarshal(input, &timeStr); err != nil {
		return err
	}

	parsedTime, err := time.Parse(TimeLayout, timeStr)
	if err == nil {
		*rt = RideTime(parsedTime.Unix())
	}
	return err
}

var (
	benefits = map[pb.EBenefitType]IDWithName{
		pb.EBenefitType_BENEFIT_TYPE_COFFEE:             {ID: 0, Name: "coffee"},
		pb.EBenefitType_BENEFIT_TYPE_CHARGER:            {ID: 1, Name: "charger"},
		pb.EBenefitType_BENEFIT_TYPE_PRESS:              {ID: 2, Name: "press"},
		pb.EBenefitType_BENEFIT_TYPE_TV:                 {ID: 3, Name: "tv"},
		pb.EBenefitType_BENEFIT_TYPE_WI_FI:              {ID: 4, Name: "wi-fi"},
		pb.EBenefitType_BENEFIT_TYPE_NO_TICKET_REQUIRED: {ID: 5, Name: "no-ticket-required"},
		pb.EBenefitType_BENEFIT_TYPE_WC:                 {ID: 6, Name: "wc"},
		pb.EBenefitType_BENEFIT_TYPE_CONDITIONER:        {ID: 7, Name: "conditioner"},
		pb.EBenefitType_BENEFIT_TYPE_COMMON_AUDIO:       {ID: 8, Name: "common-audio"},
	}
	benefitTypes = make(map[IDWithName]pb.EBenefitType)
)

func init() {
	for benefit, name := range benefits {
		benefitTypes[name] = benefit
	}
}

type RideBenefit pb.EBenefitType

func (rb RideBenefit) Unwrap() pb.EBenefitType {
	return pb.EBenefitType(rb)
}

func (rb RideBenefit) MarshalJSON() ([]byte, error) {
	protoBenefit := rb.Unwrap()
	if rawBenefit, ok := benefits[protoBenefit]; ok {
		return json.Marshal(rawBenefit)
	} else {
		return nil, fmt.Errorf("unknown EBenefitType: %v", protoBenefit)
	}
}

func (rb *RideBenefit) UnmarshalJSON(input []byte) error {
	var rawBenefit IDWithName
	if err := json.Unmarshal(input, &rawBenefit); err != nil {
		return err
	}

	protoBenefit, ok := benefitTypes[rawBenefit]
	if !ok {
		// TODO: log warning
		protoBenefit = pb.EBenefitType_BENEFIT_TYPE_UNKNOWN
	}
	*rb = RideBenefit(protoBenefit)
	return nil
}

type JSONTime time.Time

func (jsonTime JSONTime) Time() time.Time {
	return time.Time(jsonTime)
}

func (jsonTime JSONTime) MarshalJSON() ([]byte, error) {
	var t = jsonTime.Time()
	if t.IsZero() {
		return []byte(nullJSONValue), nil
	}
	return json.Marshal(t.Format(TimeLayout))
}

func (jsonTime *JSONTime) UnmarshalText(text []byte) error {
	var t, err = time.Parse(TimeLayout, string(text))
	if err == nil {
		*jsonTime = JSONTime(t)
	}
	return err
}

func buildExplanation(httpResponse *http.Response, explain []JSONExplanationCommunicationAct) *wpb.TExplanation {
	explanation := &wpb.TExplanation{}
	if httpResponse != nil {
		explanation.ConnectorRequestUrl = httpResponse.Request.URL.String()
		explanation.ConnectorResponseCode = uint32(httpResponse.StatusCode)
	}
	if explain != nil {
		explanation.CommunicationActs = make([]*wpb.TExplanationCommunicationAct, len(explain))
		for i, act := range explain {
			explanation.CommunicationActs[i] = &wpb.TExplanationCommunicationAct{
				RequestUrl:      act.RequestURL,
				RequestMethod:   act.RequestMethod,
				RequestHeaders:  fmt.Sprintf("%v", act.RequestHeaders),
				RequestBody:     fmt.Sprintf("%v", act.RequestBody),
				ResponseCode:    act.ResponseCode,
				ResponseHeaders: fmt.Sprintf("%v", act.ResponseHeaders),
				ResponseBody:    fmt.Sprintf("%v", act.ResponseBody),
			}
		}
	}
	return explanation
}

type JSONExplanationCommunicationAct struct {
	RequestURL      string            `json:"request_url"`
	RequestMethod   string            `json:"request_method"`
	RequestHeaders  map[string]string `json:"request_headers"`
	RequestBody     interface{}       `json:"request_body"`
	ResponseCode    uint32            `json:"response_code"`
	ResponseHeaders map[string]string `json:"response_headers"`
	ResponseBody    interface{}       `json:"response_body"`
}

type JSONBookParamsWithExplain struct {
	Explain []JSONExplanationCommunicationAct `json:"explain"`
	Result  JSONBookParamsMapper              `json:"result"`
}

type JSONRideWithExplain struct {
	Explain []JSONExplanationCommunicationAct `json:"explain"`
	Result  JSONRide                          `json:"result"`
}

type JSONRidesWithExplain struct {
	Explain []JSONExplanationCommunicationAct `json:"explain"`
	Result  []JSONRide                        `json:"result"`
}

type RegisterType struct {
	Title       string `json:"title"`
	Code        string `json:"code"`
	Description string `json:"description"`
}

type Carrier struct {
	RegisterNumber string       `json:"registerNumber"`
	Name           string       `json:"name"`
	Inn            string       `json:"inn"`
	FirstName      string       `json:"firstName"`
	ActualAddress  string       `json:"actualAddress"`
	MiddleName     string       `json:"middleName"`
	LastName       string       `json:"lastName"`
	RegisterType   RegisterType `json:"registerType"`
	LegalName      string       `json:"legalName"`
	Timetable      string       `json:"timetable"`
	LegalAddress   string       `json:"legalAddress"`
	ID             uint32       `json:"id"`
}

type Supplier struct {
	RegisterNumber string       `json:"registerNumber"`
	Code           string       `json:"code"`
	Name           string       `json:"name"`
	FirstName      string       `json:"firstName"`
	ActualAddress  string       `json:"actualAddress"`
	MiddleName     string       `json:"middleName"`
	LastName       string       `json:"lastName"`
	RegisterType   RegisterType `json:"registerType"`
	TaxationNumber string       `json:"taxationNumber"`
	LegalName      string       `json:"legalName"`
	Timetable      string       `json:"timetable"`
	LegalAddress   string       `json:"legalAddress"`
	ID             uint32       `json:"id"`
}

type JSONRide struct {
	SupplierRideID   string        `json:"@id,omitempty"`
	RideID           string        `json:"id"`
	Status           IDWithName    `json:"status"`
	RefundConditions string        `json:"refundConditions"`
	Number           string        `json:"number"`
	Currency         string        `json:"currency"`
	Partner          string        `json:"partner"`
	SupplierModel    *Supplier     `json:"supplierModel,omitempty"`
	Fee              float64       `json:"fee"`
	YandexFee        float64       `json:"yandexFee"`
	From             RideEndpoint  `json:"from"`
	CanPayOffline    bool          `json:"canPayOffline"`
	PartnerPhone     string        `json:"partnerPhone"`
	OnlinePrice      float64       `json:"onlinePrice"`
	Connector        string        `json:"connector"`
	To               RideEndpoint  `json:"to"`
	Carrier          string        `json:"carrier"`
	CarrierID        string        `json:"carrierID"`
	FreeSeats        int           `json:"freeSeats"`
	PartnerName      string        `json:"partnerName"`
	Bus              string        `json:"bus"`
	Price            float64       `json:"price"`
	PartnerEmail     string        `json:"partnerEmail"`
	BookOnly         bool          `json:"bookOnly"`
	Benefits         []RideBenefit `json:"benefits,omitempty"`
	Name             string        `json:"name"`
	Arrival          RideTime      `json:"arrival"`
	Departure        RideTime      `json:"departure"`
	Duration         int           `json:"duration,omitempty"`
	TicketLimit      int           `json:"ticketLimit"`
	OnlineRefund     bool          `json:"onlineRefund"`
	BookFields       []string      `json:"bookFields"`
	CarrierModel     *Carrier      `json:"carrierModel,omitempty"`
	BookURLOrig      string        `json:"book_url_orig,omitempty"`
	BookURL          string        `json:"book_url,omitempty"`
}

type RaspJSONRide struct {
	JSONRide
	From string `json:"from"`
	To   string `json:"to"`
}

type JSONBookParamsMapper struct {
	RefundConditions string `json:"refundConditions"`
	DocTypes         []struct {
		Code string     `json:"code"`
		Type IDWithName `json:"type"`
	} `json:"docTypes"`
	TicketTypes []struct {
		Price float64    `json:"price"`
		Fee   float64    `json:"fee"`
		Code  string     `json:"code"`
		Type  IDWithName `json:"type"`
	} `json:"ticketTypes"`
	GenderTypes []struct {
		Code string     `json:"code"`
		Type IDWithName `json:"type"`
	} `json:"genderTypes"`
	Seats []struct {
		Code   string `json:"code"`
		Number string `json:"number"`
	} `json:"seats"`
	Map []struct {
		X      uint32     `json:"x"`
		Y      uint32     `json:"y"`
		Type   IDWithName `json:"type"`
		Status IDWithName `json:"status"`
		Number string     `json:"number"`
	} `json:"map"`
	Citizenships []struct {
		Code string `json:"code"`
		Name string `json:"name"`
	} `json:"citizenships"`
}

type JSONCalendarItemMapper struct {
	Type                string                               `json:"type"`
	RideCount           uint32                               `json:"rideCount"`
	MinPrices           map[tpb.ECurrency]float64            `json:"minPrices"`
	MinPricesBySupplier map[string]map[tpb.ECurrency]float64 `json:"byPartner"`
}

type JSONSegments struct {
	Segments [][]string `json:"segments"`
}

type JSONSegmentsWithExplain struct {
	Explain []JSONExplanationCommunicationAct `json:"explain"`
	Result  JSONSegments                      `json:"result"`
}

type JSONPassengerMixin struct {
	FirstName  string   `json:"firstName"`
	MiddleName string   `json:"middleName"`
	LastName   string   `json:"lastName"`
	BirthDate  JSONTime `json:"birthDate"`
}

type JSONBookingPassengerMapper struct {
	JSONPassengerMixin
	Phone           string           `json:"phone"`
	Email           string           `json:"email"`
	SeatCode        string           `json:"seatCode"`
	CitizenshipCode string           `json:"citizenshipCode"`
	GenderCode      string           `json:"genderCode"`
	DocTypeCode     string           `json:"docTypeCode"`
	DocTypeID       pb.EDocumentType `json:"docTypeId"`
	DocSeries       string           `json:"docSeries"`
	DocNumber       string           `json:"docNumber"`
	TicketTypeCode  string           `json:"ticketTypeCode"`
}

type JSONBookingMapper struct {
	Passengers []JSONBookingPassengerMapper `json:"passengers"`
}

type JSONTicketPassengerMapper struct {
	JSONPassengerMixin
	Citizenship string     `json:"citizenship"`
	GenderType  IDWithName `json:"genderType"`
	DocType     IDWithName `json:"docType"`
	DocNumber   string     `json:"docNumber"`
	TicketType  IDWithName `json:"ticketType"`
	Seat        string     `json:"seat"`
}

type JSONTicketMapper struct {
	ID       string     `json:"@id"`
	Status   IDWithName `json:"status"`
	URL      string     `json:"url"`
	Price    float32    `json:"price"`
	PriceVAT string     `json:"priceVat"`
	FeeVAT   string     `json:"feeVat"`
	Revenue  *float32   `json:"revenue"`
	Data     struct {
		ID       string `json:"id"`
		Code     string `json:"code"`
		Number   string `json:"number"`
		Series   string `json:"series"`
		Barcode  string `json:"barcode"`
		Platform string `json:"platform"`
	} `json:"data"`
	Passenger JSONTicketPassengerMapper `json:"passenger"`
}

type JSONOrder struct {
	ID                 string             `json:"@id"`
	Status             IDWithName         `json:"status"`
	ExpirationDateTime string             `json:"expirationDateTime"`
	Tickets            []JSONTicketMapper `json:"tickets"`
}

type JSONOrderWithExplain struct {
	Explain []JSONExplanationCommunicationAct `json:"explain"`
	Result  JSONOrder                         `json:"result"`
}

type JSONRefundInfo struct {
	Price     float32 `json:"price"`
	Available bool    `json:"available"`
}

type JSONRefundInfoWithExplain struct {
	Explain []JSONExplanationCommunicationAct `json:"explain"`
	Result  JSONRefundInfo                    `json:"result"`
}

type JSONRefund struct {
	Price float32 `json:"price"`
}

type JSONRefundWithExplain struct {
	Explain []JSONExplanationCommunicationAct `json:"explain"`
	Result  JSONRefund                        `json:"result"`
}

type JSONErrorMapper struct {
	Error struct {
		Code    uint32 `json:"code"`
		Message string `json:"message"`
	} `json:"error"`
}
