package connector

import (
	"fmt"
	"regexp"
	"strings"
	"time"

	tpb "a.yandex-team.ru/travel/proto"
	"google.golang.org/protobuf/types/known/timestamppb"

	"a.yandex-team.ru/travel/buses/backend/internal/common/utils"
	pb "a.yandex-team.ru/travel/buses/backend/proto"
	wpb "a.yandex-team.ru/travel/buses/backend/proto/worker"
)

func (c *HTTPClient) PostBook(rideID string, contacts *wpb.TBookContacts, passengers []*wpb.TBookPassenger) (*wpb.TOrder, *wpb.TExplanation, error) {
	const funcName = "HTTPClient.PostBook"

	_, shortRideID, err := utils.ParseRideID(rideID)
	if err != nil {
		return nil, nil, fmt.Errorf("%s: %w", funcName, err)
	}

	bookingPayload := convertPassengersToBookingJSON(contacts, passengers)
	response, err := c.post(fmt.Sprintf("rides/%s/book?explain=1", shortRideID), bookingPayload, c.cfg.Timeout)
	if err != nil {
		return nil, nil, fmt.Errorf("%s: %w", funcName, err)
	}

	var orderWithExplain JSONOrderWithExplain
	defer func() { _ = response.Body.Close() }()
	if err := readJSON(response.Body, &orderWithExplain); err != nil {
		return nil, buildExplanation(response, nil), fmt.Errorf("%s: %w", funcName, err)
	}

	tOrder, err := convertOrderToProto(orderWithExplain.Result)
	return tOrder, buildExplanation(response, orderWithExplain.Explain), err
}

func convertPassengersToBookingJSON(contacts *wpb.TBookContacts, passengers []*wpb.TBookPassenger) JSONBookingMapper {
	var jsonBooking = JSONBookingMapper{
		Passengers: make([]JSONBookingPassengerMapper, len(passengers)),
	}
	for i, passenger := range passengers {
		documentSeries, documentNumber := splitDocumentNumber(passenger.DocumentType.Id, passenger.DocumentNumber)
		jsonBooking.Passengers[i] = JSONBookingPassengerMapper{
			JSONPassengerMixin: JSONPassengerMixin{
				FirstName:  passenger.FirstName,
				MiddleName: passenger.MiddleName,
				LastName:   passenger.LastName,
				BirthDate:  JSONTime(utils.ConvertProtoDateToTime(passenger.BirthDate)),
			},
			GenderCode:      passenger.GetGender().GetPartnerId(),
			DocTypeCode:     passenger.DocumentType.PartnerId,
			DocTypeID:       passenger.DocumentType.Id,
			DocSeries:       documentSeries,
			DocNumber:       documentNumber,
			CitizenshipCode: passenger.GetCitizenship().GetPartnerId(),
			SeatCode:        passenger.GetSeat().GetPartnerId(),
			TicketTypeCode:  passenger.GetTicketType().GetPartnerId(),
			Phone:           contacts.Phone,
			Email:           contacts.Email,
		}
	}
	return jsonBooking
}

var nonDigitsPattern = regexp.MustCompile(`\D`)

func splitDocumentNumber(documentType pb.EDocumentType, documentNumber string) (string, string) {
	if documentType == pb.EDocumentType_DOCUMENT_TYPE_FOREIGN_PASSPORT {
		return "", documentNumber
	}

	parts := strings.SplitN(documentNumber, " ", 2)
	if len(parts) < 2 {
		if documentType == pb.EDocumentType_DOCUMENT_TYPE_RU_PASSPORT && len(documentNumber) == 10 {
			return documentNumber[0:4], documentNumber[4:]
		}
		return "", documentNumber
	}

	return parts[0], nonDigitsPattern.ReplaceAllString(parts[1], "")
}

var ticketVatToEnum = map[string]wpb.ETicketVAT{
	"nds_none":   wpb.ETicketVAT_TV_NDS_NONE,
	"nds_0":      wpb.ETicketVAT_TV_NDS_0,
	"nds_10":     wpb.ETicketVAT_TV_NDS_10,
	"nds_10_110": wpb.ETicketVAT_TV_NDS_10_110,
	"nds_18":     wpb.ETicketVAT_TV_NDS_18,
	"nds_18_118": wpb.ETicketVAT_TV_NDS_18_118,
	"nds_20":     wpb.ETicketVAT_TV_NDS_20,
	"nds_20_120": wpb.ETicketVAT_TV_NDS_20_120,
}

func convertOrderToProto(jsonOrder JSONOrder) (*wpb.TOrder, error) {
	const funcName = "convertOrderToProto"
	var expiresAt, err = time.Parse(time.RFC3339, jsonOrder.ExpirationDateTime)
	if err != nil {
		return nil, fmt.Errorf("%s: %w", funcName, err)
	}
	var order = &wpb.TOrder{
		Id:        jsonOrder.ID,
		Status:    wpb.EOrderStatus(jsonOrder.Status.ID),
		ExpiresAt: &timestamppb.Timestamp{Seconds: expiresAt.Unix()},
		Tickets:   make([]*wpb.TTicket, len(jsonOrder.Tickets)),
	}
	for i, jsonTicket := range jsonOrder.Tickets {
		ticket := &wpb.TTicket{
			Id:        jsonTicket.ID,
			Status:    wpb.ETicketStatus(jsonTicket.Status.ID),
			BlankUrl:  jsonTicket.URL,
			Code:      jsonTicket.Data.Code,
			Series:    jsonTicket.Data.Series,
			Number:    jsonTicket.Data.Number,
			Barcode:   jsonTicket.Data.Barcode,
			Platform:  jsonTicket.Data.Platform,
			Price:     makePrice(jsonTicket.Price),
			PriceVat:  ticketVatToEnum[jsonTicket.PriceVAT],
			FeeVat:    ticketVatToEnum[jsonTicket.FeeVAT],
			Passenger: convertPassengerToProto(jsonTicket.Passenger),
		}
		if jsonTicket.Revenue != nil {
			ticket.Revenue = makePrice(*jsonTicket.Revenue)
		}
		order.Tickets[i] = ticket
	}
	return order, nil
}

func makePrice(amount float32) *tpb.TPrice {
	return &tpb.TPrice{
		Amount:    int64(amount * 100),
		Currency:  tpb.ECurrency_C_RUB,
		Precision: pricePrecision,
	}
}

func convertPassengerToProto(jsonPassenger JSONTicketPassengerMapper) *wpb.TTicketPassenger {
	passenger := &wpb.TTicketPassenger{
		FirstName:  jsonPassenger.FirstName,
		LastName:   jsonPassenger.LastName,
		TicketType: pb.ETicketType(jsonPassenger.TicketType.ID),
		Seat:       jsonPassenger.Seat,
	}
	if !jsonPassenger.BirthDate.Time().IsZero() {
		passenger.BirthDate = utils.ConvertTimeToProtoDate(jsonPassenger.BirthDate.Time())
	}
	if !jsonPassenger.GenderType.IsEmpty() {
		passenger.OptionalGender = &wpb.TTicketPassenger_Gender{Gender: pb.EGenderType(jsonPassenger.GenderType.ID)}
	}
	if jsonPassenger.Citizenship != "" {
		passenger.OptionalCitizenship = &wpb.TTicketPassenger_Citizenship{Citizenship: jsonPassenger.Citizenship}
	}
	if !jsonPassenger.DocType.IsEmpty() {
		passenger.OptionalDocumentType = &wpb.TTicketPassenger_DocumentType{DocumentType: pb.EDocumentType(jsonPassenger.DocType.ID)}
	}
	if jsonPassenger.DocNumber != "" {
		passenger.OptionalDocumentNumber = &wpb.TTicketPassenger_DocumentNumber{DocumentNumber: jsonPassenger.DocNumber}
	}
	if jsonPassenger.MiddleName != "" {
		passenger.OptionalMiddleName = &wpb.TTicketPassenger_MiddleName{MiddleName: jsonPassenger.MiddleName}
	}
	return passenger
}
