package connector

import (
	"encoding/json"
	"fmt"
	"io/ioutil"
	"strings"

	tpb "a.yandex-team.ru/travel/proto"

	"a.yandex-team.ru/travel/buses/backend/internal/common/utils"
	pb "a.yandex-team.ru/travel/buses/backend/proto"
	wpb "a.yandex-team.ru/travel/buses/backend/proto/worker"
)

func (c *HTTPClient) GetBookParams(rideID string) (*pb.TBookParams, *pb.TRideRefinement, *wpb.TExplanation, error) {
	const funcName = "Connector.GetBookParams"

	_, shortRideID, err := utils.ParseRideID(rideID)
	if err != nil {
		return nil, nil, nil, fmt.Errorf("%s: %w", funcName, err)
	}

	response, err := c.get(fmt.Sprintf("rides/%s/book-params?explain=1", shortRideID), c.cfg.Timeout)
	if err != nil {
		return nil, nil, nil, fmt.Errorf("%s: %w", funcName, err)
	}

	defer func() { _ = response.Body.Close() }()
	responseBytes, err := ioutil.ReadAll(response.Body)
	if err != nil {
		return nil, nil, buildExplanation(response, nil), fmt.Errorf("%s: %w", funcName, err)
	}

	var jsonBookParamsWithExplain JSONBookParamsWithExplain
	if err = json.Unmarshal(responseBytes, &jsonBookParamsWithExplain); err != nil {
		return nil, nil, buildExplanation(response, nil), fmt.Errorf("%s: %w", funcName, err)
	}

	var jsonRideWithExplain JSONRideWithExplain
	if err = json.Unmarshal(responseBytes, &jsonRideWithExplain); err != nil {
		return nil, nil, buildExplanation(response, jsonBookParamsWithExplain.Explain), fmt.Errorf("%s: %w", funcName, err)
	}

	return convertBookParamsToProto(jsonBookParamsWithExplain.Result),
		convertRideRefinementToProto(jsonRideWithExplain.Result),
		buildExplanation(response, jsonBookParamsWithExplain.Explain),
		nil
}

func convertRideRefinementToProto(rideRefinement JSONRide) *pb.TRideRefinement {
	protoRideRefinement := pb.TRideRefinement{
		DepartureTime:    int64(rideRefinement.Departure),
		ArrivalTime:      int64(rideRefinement.Arrival),
		FromDesc:         rideRefinement.From.Desc,
		ToDesc:           rideRefinement.To.Desc,
		RefundConditions: rideRefinement.RefundConditions,
	}

	if pk, err := utils.LoadPointKey(rideRefinement.From.ID); err == nil {
		protoRideRefinement.From = pk
	}
	if pk, err := utils.LoadPointKey(rideRefinement.To.ID); err == nil {
		protoRideRefinement.To = pk
	}

	if rideRefinement.Price != 0 {
		protoRideRefinement.Price = &tpb.TPrice{
			Amount:    int64(rideRefinement.Price * 100),
			Currency:  priceCurrency,
			Precision: pricePrecision,
		}
	}
	if rideRefinement.Fee != 0 {
		protoRideRefinement.Fee = &tpb.TPrice{
			Amount:    int64(rideRefinement.Fee * 100),
			Currency:  priceCurrency,
			Precision: pricePrecision,
		}
	}

	return &protoRideRefinement
}

func convertBookParamsToProto(bookParams JSONBookParamsMapper) *pb.TBookParams {
	protoBookParams := pb.TBookParams{
		Documents:    make([]*pb.TDocumentType, len(bookParams.DocTypes)),
		TicketTypes:  make([]*pb.TTicketType, len(bookParams.TicketTypes)),
		Genders:      make([]*pb.TGender, len(bookParams.GenderTypes)),
		Seats:        make([]*pb.TSeat, len(bookParams.Seats)),
		PlacesMap:    make([]*pb.TPlace, len(bookParams.Map)),
		Citizenships: make([]*pb.TCitizenship, len(bookParams.Citizenships)),
	}
	for i, document := range bookParams.DocTypes {
		protoBookParams.Documents[i] = &pb.TDocumentType{
			Id:        pb.EDocumentType(document.Type.ID),
			PartnerId: document.Code,
		}
	}
	for i, ticketType := range bookParams.TicketTypes {
		protoBookParams.TicketTypes[i] = &pb.TTicketType{
			Id:        pb.ETicketType(ticketType.Type.ID),
			PartnerId: ticketType.Code,
			Price: &tpb.TPrice{
				Amount:    int64(ticketType.Price * 100),
				Currency:  priceCurrency,
				Precision: pricePrecision,
			},
			PartnerFee: &tpb.TPrice{
				Amount:    int64(ticketType.Fee * 100),
				Currency:  priceCurrency,
				Precision: pricePrecision,
			},
		}
	}
	for i, gender := range bookParams.GenderTypes {
		protoBookParams.Genders[i] = &pb.TGender{
			Id:        pb.EGenderType(gender.Type.ID),
			PartnerId: gender.Code,
		}
	}
	for i, seat := range bookParams.Seats {
		protoBookParams.Seats[i] = &pb.TSeat{
			Id:        seat.Number,
			PartnerId: seat.Code,
		}
	}
	for i, place := range bookParams.Map {
		number := place.Number
		if strings.ToLower(number) == "z" || strings.ToLower(number) == "v" {
			number = "×" // TODO: move it to connectors: https://st.yandex-team.ru/BUSES-1654
		}
		protoBookParams.PlacesMap[i] = &pb.TPlace{
			Id:     number,
			X:      place.X,
			Y:      place.Y,
			Type:   pb.EPlaceType(place.Type.ID),
			Status: pb.EPlaceStatus(place.Status.ID),
		}
	}
	for i, citizenship := range bookParams.Citizenships {
		protoBookParams.Citizenships[i] = &pb.TCitizenship{
			Id:        citizenship.Name,
			PartnerId: citizenship.Code,
		}
	}
	return &protoBookParams
}
