package connector

import (
	"bytes"
	"encoding/json"
	"fmt"
	"net/http"
	"time"

	"a.yandex-team.ru/library/go/core/log/zap"
	"a.yandex-team.ru/library/go/httputil/headers"
	tpb "a.yandex-team.ru/travel/proto"

	"a.yandex-team.ru/travel/buses/backend/internal/common/connector/mock"
	"a.yandex-team.ru/travel/buses/backend/internal/common/dict"
	"a.yandex-team.ru/travel/buses/backend/internal/common/logging"
	pb "a.yandex-team.ru/travel/buses/backend/proto"
	wpb "a.yandex-team.ru/travel/buses/backend/proto/worker"
)

type Config struct {
	APIURL          string        `config:"connector-apiurl,required"`
	Timeout         time.Duration `config:"connector-timeout,required"`
	SegmentsTimeout time.Duration `config:"connector-segmentstimeout,required"`
}

var DefaultConfig = Config{
	Timeout:         1 * time.Minute,
	SegmentsTimeout: 15 * time.Minute,
}

var MockedConfig = Config{}

type Client interface {
	GetSegments() ([]*wpb.TSegment, *wpb.TExplanation, error)
	GetSearch(from *pb.TPointKey, to *pb.TPointKey, date *tpb.TDate, tryNoCache bool) ([]*pb.TRide, *wpb.TExplanation, error)
	GetBookParams(rideID string) (*pb.TBookParams, *pb.TRideRefinement, *wpb.TExplanation, error)
	PostBook(rideID string, contacts *wpb.TBookContacts, passengers []*wpb.TBookPassenger) (*wpb.TOrder, *wpb.TExplanation, error)
	PostConfirm(orderID string) (*wpb.TOrder, *wpb.TExplanation, error)
	GetRefundInfo(ticketID string) (*wpb.TRefundInfo, *wpb.TExplanation, error)
	PostRefund(ticketID string) (*wpb.TRefund, *wpb.TExplanation, error)
}

type HTTPClient struct {
	cfg       *Config
	logger    *zap.Logger
	transport http.RoundTripper
	supplier  dict.Supplier
}

func (c *HTTPClient) get(path string, timeout time.Duration) (*http.Response, error) {
	client := http.Client{Timeout: timeout, Transport: c.transport}
	response, err := client.Get(
		fmt.Sprintf("%s/%s/%s", c.cfg.APIURL, c.supplier.Name, path))

	if err != nil {
		return nil, NewErrUnavailable(err)
	}

	if err := errorFromResponse(response); err != nil {
		return nil, err
	}

	return response, nil
}

func (c *HTTPClient) post(path string, jsonBody interface{}, timeout time.Duration) (*http.Response, error) {
	const funcName = "HTTPClient.Post"

	var (
		content     []byte
		contentType = headers.TypeApplicationJSON
	)
	if jsonBody != nil {
		var err error
		content, err = json.Marshal(jsonBody)
		if err != nil {
			return nil, fmt.Errorf("%s: %w", funcName, err)
		}
	} else {
		contentType = headers.TypeTextPlain
	}

	var (
		client        = http.Client{Timeout: timeout, Transport: c.transport}
		response, err = client.Post(
			fmt.Sprintf("%s/%s/%s", c.cfg.APIURL, c.supplier.Name, path),
			contentType.String(),
			bytes.NewReader(content),
		)
	)

	if err != nil {
		return nil, NewErrUnavailable(err)
	}

	if err := errorFromResponse(response); err != nil {
		return nil, err
	}

	return response, nil
}

func NewClient(cfg *Config, supplierID uint32, logger *zap.Logger) (Client, error) {
	return NewClientWithTransport(cfg, supplierID, logger, nil)
}

func NewClientWithTransport(cfg *Config, supplierID uint32, logger *zap.Logger, transport http.RoundTripper) (Client, error) {
	supplier, err := dict.GetSupplier(supplierID)
	if err != nil {
		return nil, fmt.Errorf("NewClientWithTransport: %w", err)
	}
	if *cfg == MockedConfig {
		return mock.NewMockedClient(supplierID, logger)
	}
	return &HTTPClient{
		cfg:       cfg,
		logger:    logging.WithSupplierContext(logger, supplier.ID),
		supplier:  supplier,
		transport: transport,
	}, nil
}
