package mock

import (
	"fmt"
	"net/http"

	"github.com/stretchr/testify/mock"

	"a.yandex-team.ru/library/go/core/log/zap"
	pbTravel "a.yandex-team.ru/travel/proto"

	"a.yandex-team.ru/travel/buses/backend/internal/api/cache"
	"a.yandex-team.ru/travel/buses/backend/internal/common/dict"
	"a.yandex-team.ru/travel/buses/backend/internal/common/logging"
	pb "a.yandex-team.ru/travel/buses/backend/proto"
	wpb "a.yandex-team.ru/travel/buses/backend/proto/worker"
)

const MockedAPIURL = "mocked"

type SearchScenario struct {
	cases         map[cache.SearchKey][]*pb.TRide
	defaultResult []*pb.TRide
}

func newSearchScenario() *SearchScenario {
	return &SearchScenario{
		cases:         make(map[cache.SearchKey][]*pb.TRide),
		defaultResult: nil,
	}
}

func (ss *SearchScenario) SetDefault(result []*pb.TRide) {
	ss.defaultResult = result
}

func (ss *SearchScenario) Add(supplierID uint32, from *pb.TPointKey, to *pb.TPointKey,
	date *pbTravel.TDate, result []*pb.TRide) {
	key := cache.NewSearchKey(supplierID, from, to, date)
	ss.cases[key] = result
}

func (ss *SearchScenario) Get(
	supplierID uint32, from *pb.TPointKey, to *pb.TPointKey, date *pbTravel.TDate, tryNoCache bool,
) ([]*pb.TRide, *wpb.TExplanation, bool) {
	key := cache.NewSearchKey(supplierID, from, to, date)
	result, ok := ss.cases[key]
	if !ok && ss.defaultResult != nil {
		return ss.defaultResult, nil, true
	}
	return result, nil, ok
}

func (ss *SearchScenario) Clear() {
	ss.cases = make(map[cache.SearchKey][]*pb.TRide)
	ss.defaultResult = nil
}

var searchScenario = newSearchScenario()

func GetSearchScenario() *SearchScenario {
	return searchScenario
}

type MockedClient struct {
	logger   *zap.Logger
	supplier dict.Supplier
}

type ClientMock struct {
	mock.Mock
}

var clientMock *ClientMock

func SetupClientMock() (*ClientMock, func()) {
	clientMock = &ClientMock{}
	return clientMock, func() {
		clientMock = nil
	}
}

func (c *ClientMock) GetSegments() ([]*wpb.TSegment, *wpb.TExplanation, error) {
	args := c.MethodCalled("GetSegments")
	return args.Get(0).([]*wpb.TSegment), args.Get(1).(*wpb.TExplanation), args.Error(2)
}

func (c *ClientMock) GetBookParams(rideID string) (*pb.TBookParams, *pb.TRideRefinement, *wpb.TExplanation, error) {
	args := c.MethodCalled("GetBookParams", rideID)
	return args.Get(0).(*pb.TBookParams), args.Get(1).(*pb.TRideRefinement), args.Get(2).(*wpb.TExplanation), args.Error(3)

}

func (c *ClientMock) PostBook(rideID string, contacts *wpb.TBookContacts, passengers []*wpb.TBookPassenger) (*wpb.TOrder, *wpb.TExplanation, error) {
	args := c.MethodCalled("PostBook", rideID, contacts, passengers)
	return args.Get(0).(*wpb.TOrder), args.Get(1).(*wpb.TExplanation), args.Error(2)
}

func (c *ClientMock) PostConfirm(orderID string) (*wpb.TOrder, *wpb.TExplanation, error) {
	args := c.MethodCalled("PostConfirm", orderID)
	return args.Get(0).(*wpb.TOrder), args.Get(1).(*wpb.TExplanation), args.Error(2)
}

func (c *ClientMock) GetRefundInfo(ticketID string) (*wpb.TRefundInfo, *wpb.TExplanation, error) {
	args := c.MethodCalled("GetRefundInfo", ticketID)
	return args.Get(0).(*wpb.TRefundInfo), args.Get(1).(*wpb.TExplanation), args.Error(2)
}

func (c *ClientMock) PostRefund(ticketID string) (*wpb.TRefund, *wpb.TExplanation, error) {
	args := c.MethodCalled("PostRefund", ticketID)
	return args.Get(0).(*wpb.TRefund), args.Get(1).(*wpb.TExplanation), args.Error(2)
}

func NewMockedClient(supplierID uint32, logger *zap.Logger) (*MockedClient, error) {
	supplier, err := dict.GetSupplier(supplierID)
	if err != nil {
		return nil, fmt.Errorf("NewClient: %w", err)
	}
	return &MockedClient{
		logger:   logging.WithSupplierContext(logger, supplier.ID),
		supplier: supplier,
	}, nil
}

func (c *MockedClient) GetSegments() ([]*wpb.TSegment, *wpb.TExplanation, error) {
	if clientMock != nil {
		return clientMock.GetSegments()
	}
	return nil, nil, fmt.Errorf("not implemented")
}

func (c *MockedClient) GetSearch(
	from *pb.TPointKey, to *pb.TPointKey, date *pbTravel.TDate, tryNoCache bool,
) ([]*pb.TRide, *wpb.TExplanation, error) {
	const logMessage = "MockedClient.GetSearch"
	searchScenario := GetSearchScenario()
	result, explanation, ok := searchScenario.Get(c.supplier.ID, from, to, date, tryNoCache)
	if !ok {
		return nil, nil, fmt.Errorf("%s: no scenario found for %s, %s, %s, %s",
			logMessage, c.supplier.Name, from.String(), to.String(), date.String())
	}
	return result, explanation, nil
}

func (c *MockedClient) GetBookParams(rideID string) (*pb.TBookParams, *pb.TRideRefinement, *wpb.TExplanation, error) {
	if clientMock != nil {
		return clientMock.GetBookParams(rideID)
	}
	return nil, nil, nil, fmt.Errorf("not implemented")
}

func (c *MockedClient) PostBook(rideID string, contacts *wpb.TBookContacts, passengers []*wpb.TBookPassenger) (*wpb.TOrder, *wpb.TExplanation, error) {
	if clientMock != nil {
		return clientMock.PostBook(rideID, contacts, passengers)
	}
	return nil, nil, fmt.Errorf("not implemented")
}

func (c *MockedClient) PostConfirm(orderID string) (*wpb.TOrder, *wpb.TExplanation, error) {
	if clientMock != nil {
		return clientMock.PostConfirm(orderID)
	}
	return nil, nil, fmt.Errorf("not implemented")
}

func (c *MockedClient) GetRefundInfo(ticketID string) (*wpb.TRefundInfo, *wpb.TExplanation, error) {
	if clientMock != nil {
		return clientMock.GetRefundInfo(ticketID)
	}
	return nil, nil, fmt.Errorf("not implemented")
}

func (c *MockedClient) PostRefund(ticketID string) (*wpb.TRefund, *wpb.TExplanation, error) {
	if clientMock != nil {
		return clientMock.PostRefund(ticketID)
	}
	return nil, nil, fmt.Errorf("not implemented")
}

type TransportMock func(req *http.Request) *http.Response

func (m TransportMock) RoundTrip(req *http.Request) (*http.Response, error) {
	return m(req), nil
}
