package connector

import (
	"fmt"

	"a.yandex-team.ru/library/go/core/log"
	tpb "a.yandex-team.ru/travel/proto"

	"a.yandex-team.ru/travel/buses/backend/internal/common/dict"
	"a.yandex-team.ru/travel/buses/backend/internal/common/utils"
	pb "a.yandex-team.ru/travel/buses/backend/proto"
	wpb "a.yandex-team.ru/travel/buses/backend/proto/worker"
)

func (c *HTTPClient) GetSearch(from *pb.TPointKey, to *pb.TPointKey, date *tpb.TDate, tryNoCache bool) ([]*pb.TRide, *wpb.TExplanation, error) {
	const funcName = "HTTPClient.GetSearch"

	fromStr, _ := utils.DumpPointKey(from)
	toStr, _ := utils.DumpPointKey(to)
	response, err := c.get(
		fmt.Sprintf("rides/search?from-id=%s&to-id=%s&date=%s&explain=1&try-no-cache=%t",
			fromStr, toStr, utils.DumpDate(date), tryNoCache),
		c.cfg.Timeout,
	)
	if err != nil {
		return nil, nil, fmt.Errorf("%s: %w", funcName, err)
	}

	defer func() { _ = response.Body.Close() }()
	var jsonRidesWithExplain JSONRidesWithExplain
	if err := readJSON(response.Body, &jsonRidesWithExplain); err != nil {
		return nil, buildExplanation(response, nil), fmt.Errorf("json unmarshal error: %w", err)
	}

	ridesProto := convertRidesToProto(c.supplier.ID, jsonRidesWithExplain.Result, c.logger)
	return ridesProto, buildExplanation(response, jsonRidesWithExplain.Explain), nil
}

var registerCodeToRegisterType = map[string]pb.ERegisterType{
	"ОГРН":   pb.ERegisterType_REGISTER_TYPE_COMPANY,
	"ОГРНИП": pb.ERegisterType_REGISTER_TYPE_ENTREPRENEUR,
}

func registerTypeByCode(code string) pb.ERegisterType {
	registerType, ok := registerCodeToRegisterType[code]
	if !ok {
		return pb.ERegisterType_REGISTER_TYPE_UNKNOWN
	}
	return registerType
}

func carrierToProto(carrier *Carrier) *pb.TCarrier {
	if carrier == nil {
		return nil
	}
	return &pb.TCarrier{
		Name:           carrier.Name,
		LegalName:      carrier.LegalName,
		RegisterNumber: carrier.RegisterNumber,
		LegalAddress:   carrier.LegalAddress,
		ActualAddress:  carrier.ActualAddress,
		Inn:            carrier.Inn,
		RegisterType:   registerTypeByCode(carrier.RegisterType.Code),
		Timetable:      carrier.Timetable,
		LastName:       carrier.LastName,
		MiddleName:     carrier.MiddleName,
		FirstName:      carrier.FirstName,
		ID:             carrier.ID,
	}
}

func supplierToProto(supplier *Supplier) *pb.TSupplier {
	if supplier == nil {
		return nil
	}
	return &pb.TSupplier{
		Name:           supplier.Name,
		LegalName:      supplier.LegalName,
		RegisterNumber: supplier.RegisterNumber,
		LegalAddress:   supplier.LegalAddress,
		ActualAddress:  supplier.ActualAddress,
		Inn:            supplier.TaxationNumber,
		RegisterType:   registerTypeByCode(supplier.RegisterType.Code),
		Timetable:      supplier.Timetable,
		LastName:       supplier.LastName,
		MiddleName:     supplier.MiddleName,
		FirstName:      supplier.FirstName,
		ID:             supplier.ID,
	}
}

func benefitsToProto(benefits []RideBenefit) []pb.EBenefitType {
	protoBenefits := make([]pb.EBenefitType, 0, len(benefits))
	for _, benefit := range benefits {
		protoBenefit := benefit.Unwrap()
		if protoBenefit != pb.EBenefitType_BENEFIT_TYPE_UNKNOWN {
			protoBenefits = append(protoBenefits, protoBenefit)
		}
	}
	return protoBenefits
}

func convertRidesToProto(sID uint32, rides []JSONRide, logger log.Logger) []*pb.TRide {
	protoRides := make([]*pb.TRide, len(rides))
	converted := 0
	for _, ride := range rides {
		from, _ := utils.LoadPointKey(ride.From.ID)
		to, _ := utils.LoadPointKey(ride.To.ID)
		supplier, _ := dict.GetSupplier(sID)
		protoRide := pb.TRide{
			Id:             utils.MakeRideID(supplier, ride.SupplierRideID),
			SupplierId:     supplier.ID,
			Supplier:       supplierToProto(ride.SupplierModel),
			CarrierCode:    ride.CarrierID,
			CarrierName:    ride.Carrier,
			Carrier:        carrierToProto(ride.CarrierModel),
			From:           from,
			FromDesc:       ride.From.Desc,
			FromSupplierId: ride.From.SupplierID,
			To:             to,
			ToDesc:         ride.To.Desc,
			ToSupplierId:   ride.To.SupplierID,
			ArrivalTime:    int64(ride.Arrival),
			DepartureTime:  int64(ride.Departure),
			Price: &tpb.TPrice{
				Amount:    int64(ride.Price * 100),
				Currency:  priceCurrency,
				Precision: pricePrecision,
			},
			Fee: &tpb.TPrice{
				Amount:    int64(ride.Fee * 100),
				Currency:  priceCurrency,
				Precision: pricePrecision,
			},
			Status:           pb.ERideStatus(ride.Status.ID),
			FreeSeats:        int32(ride.FreeSeats),
			Bus:              ride.Bus,
			CanPayOffline:    ride.CanPayOffline,
			BookOnly:         ride.BookOnly,
			Benefits:         benefitsToProto(ride.Benefits),
			RouteName:        ride.Name,
			RouteNumber:      ride.Number,
			TicketLimit:      uint32(ride.TicketLimit),
			OnlineRefund:     ride.OnlineRefund,
			RefundConditions: ride.RefundConditions,
			BookFields:       ride.BookFields,
		}
		protoRides[converted] = &protoRide
		converted++
	}
	if converted != len(rides) {
		logger.Errorf("convertRidesToProto: %d rides with errors", len(rides)-converted)
		return protoRides[:converted]
	}
	return protoRides
}
