package rasp

import (
	"fmt"
	"io"
	"path"
	"sync"

	"a.yandex-team.ru/library/go/core/log/zap"
	"a.yandex-team.ru/travel/library/go/dicts/base"
	"a.yandex-team.ru/travel/library/go/dicts/repository"
	"a.yandex-team.ru/travel/proto/dicts/rasp"
	"github.com/golang/protobuf/proto"

	"a.yandex-team.ru/travel/buses/backend/internal/common/logging"
)

type Config struct {
	ResourceDir  string `config:"rasp-resourcedir,required"`
	TimeZoneFN   string `config:"rasp-timezonefn,required"`
	SettlementFN string `config:"rasp-settlementfn,required"`
	DistrictFN   string `config:"rasp-districtfn,required"`
	StationFN    string `config:"rasp-stationfn,required"`
	RegionFN     string `config:"rasp-regionfn,required"`
	CountryFN    string `config:"rasp-countryfn,required"`
}

var DefaultConfig = Config{
	ResourceDir:  "/app/rasp_data",
	TimeZoneFN:   "timezone.data",
	SettlementFN: "settlement.data",
	StationFN:    "station.data",
	DistrictFN:   "district.data",
	RegionFN:     "region.data",
	CountryFN:    "country.data",
}

type DictRepo struct {
	cfg            *Config
	logger         *zap.Logger
	mutex          sync.RWMutex
	timeZoneRepo   *repository.TimeZoneRepository
	settlementRepo *repository.SettlementRepository
	stationRepo    *repository.StationRepository
	districtRepo   *repository.DistrictRepository
	regionsRepo    *repository.RegionRepository
	countryRepo    *repository.CountryRepository
	geoIDIndex     *map[int32]int32
}

type settlementWriter struct {
	geoIDIndex     *map[int32]int32
	settlementRepo *repository.SettlementRepository
}

func (sw *settlementWriter) Write(b []byte) (int, error) {
	settlement := &rasp.TSettlement{}
	if err := proto.Unmarshal(b, settlement); err != nil {
		return 0, fmt.Errorf("settlementWriter:Write: %w", err)
	}
	n, err := sw.settlementRepo.Write(b)
	if err != nil {
		return 0, err
	}
	(*sw.geoIDIndex)[settlement.GeoId] = settlement.Id
	return n, nil
}

func NewRepo(cfg *Config, logger *zap.Logger) *DictRepo {
	geoIDIndex := make(map[int32]int32)
	return &DictRepo{
		cfg:            cfg,
		logger:         logging.WithModuleContext(logger, "dict.rasp.DictRepo"),
		timeZoneRepo:   repository.NewTimeZoneRepository(),
		settlementRepo: repository.NewSettlementRepository(),
		stationRepo:    repository.NewStationRepository(),
		districtRepo:   repository.NewDistrictRepository(),
		regionsRepo:    repository.NewRegionRepository(),
		countryRepo:    repository.NewCountryRepository(),
		geoIDIndex:     &geoIDIndex,
	}
}

func (r *DictRepo) Load() error {
	const logMessage = "DictRepo.Load"

	geoIDIndex := make(map[int32]int32)
	timeZoneRepo := repository.NewTimeZoneRepository()
	settlementRepo := repository.NewSettlementRepository()
	stationRepo := repository.NewStationRepository()
	districtRepo := repository.NewDistrictRepository()
	regionsRepo := repository.NewRegionRepository()
	countryRepo := repository.NewCountryRepository()

	if err := r.load(timeZoneRepo, r.cfg.TimeZoneFN); err != nil {
		return err
	}
	if err := r.load(&settlementWriter{geoIDIndex: &geoIDIndex, settlementRepo: settlementRepo}, r.cfg.SettlementFN); err != nil {
		return err
	}
	if err := r.load(stationRepo, r.cfg.StationFN); err != nil {
		return err
	}
	if err := r.load(districtRepo, r.cfg.DistrictFN); err != nil {
		return err
	}
	if err := r.load(regionsRepo, r.cfg.RegionFN); err != nil {
		return err
	}
	if err := r.load(countryRepo, r.cfg.CountryFN); err != nil {
		return err
	}

	r.mutex.Lock()
	r.geoIDIndex = &geoIDIndex
	r.timeZoneRepo = timeZoneRepo
	r.settlementRepo = settlementRepo
	r.stationRepo = stationRepo
	r.regionsRepo = regionsRepo
	r.countryRepo = countryRepo
	r.mutex.Unlock()

	r.logger.Infof("%s: all rasp dictionaries loaded", logMessage)
	return nil
}

func (r *DictRepo) load(repo io.Writer, fn string) error {
	const logMessage = "DictRepo.load"
	bytesIterator, err := base.BuildIteratorFromFile(path.Join(r.cfg.ResourceDir, fn))
	if err != nil {
		return fmt.Errorf("%s: %w", logMessage, err)
	}
	err = bytesIterator.Populate(repo)
	if err != nil {
		return fmt.Errorf("%s: %w", logMessage, err)
	}
	r.logger.Infof("%s: rasp dictionary %s loaded", logMessage, fn)
	return nil
}

func (r *DictRepo) GetSettlement(id int32) (*rasp.TSettlement, bool) {
	r.mutex.RLock()
	defer r.mutex.RUnlock()
	return r.settlementRepo.Get(int(id))
}

func (r *DictRepo) GetSettlementByGeoID(geoID int32) (*rasp.TSettlement, bool) {
	r.mutex.RLock()
	defer r.mutex.RUnlock()
	id, ok := (*r.geoIDIndex)[geoID]
	if !ok {
		return nil, false
	}
	return r.GetSettlement(id)
}

func (r *DictRepo) GetStation(id int32) (*rasp.TStation, bool) {
	r.mutex.RLock()
	defer r.mutex.RUnlock()
	return r.stationRepo.Get(int(id))
}

func (r *DictRepo) GetTimeZone(id int32) (*rasp.TTimeZone, bool) {
	r.mutex.RLock()
	defer r.mutex.RUnlock()
	return r.timeZoneRepo.Get(int(id))
}

func (r *DictRepo) GetDistrict(id int32) (*rasp.TDistrict, bool) {
	r.mutex.RLock()
	defer r.mutex.RUnlock()
	return r.districtRepo.Get(int(id))
}

func (r *DictRepo) GetRegion(id int32) (*rasp.TRegion, bool) {
	r.mutex.RLock()
	defer r.mutex.RUnlock()
	return r.regionsRepo.Get(int(id))
}
func (r *DictRepo) GetCountry(id int32) (*rasp.TCountry, bool) {
	r.mutex.RLock()
	defer r.mutex.RUnlock()
	return r.countryRepo.Get(id)
}
