package dict

import (
	"fmt"
	"os"
)

type Supplier struct {
	ID             uint32
	Name           string
	SearchRPS      float64
	Concurrency    uint32
	Priority       uint32 // 0 is highest priority for easy adding a new one
	CalendarWarmer bool
}

// https://wiki.yandex-team.ru/travel/buses/specification/#m-ogranichenijapartnjorov

var suppliers = func() []Supplier {
	yEnvType := os.Getenv("YENV_TYPE")
	if yEnvType == "production" || yEnvType == "prestable" {
		return []Supplier{
			{ID: 13, Name: "atlasbus", SearchRPS: 5, Concurrency: 5, Priority: 9, CalendarWarmer: true},
			{ID: 11, Name: "busfor", SearchRPS: 10, Concurrency: 10, Priority: 8, CalendarWarmer: true},
			{ID: 10, Name: "ecolines", SearchRPS: 10, Concurrency: 10, Priority: 6, CalendarWarmer: true},
			{ID: 3, Name: "etraffic", SearchRPS: 5, Concurrency: 5, Priority: 3, CalendarWarmer: true},
			{ID: 7, Name: "noy", SearchRPS: 1, Concurrency: 5, Priority: 7, CalendarWarmer: true},
			{ID: 5, Name: "ok", SearchRPS: 5, Concurrency: 5, Priority: 2, CalendarWarmer: true},
			{ID: 4, Name: "ruset", SearchRPS: 5, Concurrency: 5, Priority: 0, CalendarWarmer: true},
			{ID: 9, Name: "sks", SearchRPS: 5, Concurrency: 5, Priority: 5, CalendarWarmer: true},
			{ID: 6, Name: "unitiki-new", SearchRPS: 5, Concurrency: 5, Priority: 1, CalendarWarmer: true},
			{ID: 8, Name: "yugavtotrans", SearchRPS: 2, Concurrency: 2, Priority: 4, CalendarWarmer: true},
		}
	}
	return []Supplier{
		{ID: 83, Name: "atlasbus", SearchRPS: 2, Concurrency: 2, Priority: 9, CalendarWarmer: true},
		{ID: 80, Name: "busfor", SearchRPS: 3, Concurrency: 2, Priority: 8, CalendarWarmer: true},
		{ID: 17, Name: "ecolines", SearchRPS: 3, Concurrency: 2, Priority: 6, CalendarWarmer: true},
		{ID: 10, Name: "etraffic", SearchRPS: 3, Concurrency: 2, Priority: 3, CalendarWarmer: true},
		{ID: 14, Name: "noy", SearchRPS: 3, Concurrency: 2, Priority: 7, CalendarWarmer: true},
		{ID: 12, Name: "ok", SearchRPS: 3, Concurrency: 2, Priority: 2, CalendarWarmer: true},
		{ID: 11, Name: "ruset", SearchRPS: 3, Concurrency: 2, Priority: 0, CalendarWarmer: true},
		{ID: 16, Name: "sks", SearchRPS: 3, Concurrency: 2, Priority: 5, CalendarWarmer: true},
		{ID: 13, Name: "unitiki-new", SearchRPS: 3, Concurrency: 2, Priority: 1, CalendarWarmer: true},
		{ID: 15, Name: "yugavtotrans", SearchRPS: 3, Concurrency: 2, Priority: 4, CalendarWarmer: true},
	}
}()

var suppliersMap = func() map[uint32]Supplier {
	sm := make(map[uint32]Supplier)
	for _, supplier := range suppliers {
		sm[supplier.ID] = supplier
	}
	return sm
}()

func GetSupplier(id uint32) (Supplier, error) {
	supplier, ok := suppliersMap[id]
	if !ok {
		return Supplier{}, fmt.Errorf("no such supplier with id = %d", id)
	}
	return supplier, nil
}

func GetSupplierByName(name string) (Supplier, error) {
	for _, s := range suppliers {
		if s.Name == name {
			return s, nil
		}
	}
	return Supplier{}, fmt.Errorf("no such supplier with name = %s", name)
}

func GetSuppliersList() []uint32 {
	ids := make([]uint32, len(suppliersMap))
	i := 0
	for _, supplier := range suppliers {
		ids[i] = supplier.ID
		i++
	}
	return ids
}
