package grpc

import (
	"context"
	"fmt"
	"net"

	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/library/go/core/log/zap"
	"a.yandex-team.ru/library/go/core/metrics"
	grpcMetrics "a.yandex-team.ru/travel/library/go/grpcutil/metrics"
	"a.yandex-team.ru/travel/library/go/tvm"
	middleware "github.com/grpc-ecosystem/go-grpc-middleware"
	grpcLog "github.com/grpc-ecosystem/go-grpc-middleware/logging/zap"
	recovery "github.com/grpc-ecosystem/go-grpc-middleware/recovery"
	grpcOpentracing "github.com/grpc-ecosystem/go-grpc-middleware/tracing/opentracing"
	"google.golang.org/grpc"
	"google.golang.org/grpc/codes"
	"google.golang.org/grpc/connectivity"
	"google.golang.org/grpc/status"
	"google.golang.org/grpc/test/bufconn"
)

type (
	clientTvmConfig struct {
		SelfID uint32 `yaml:"self-id"`
		ID     uint32
	}
	ClientConfig struct {
		Address string `config:"grpc-client-address,required"`
		Tvm     *clientTvmConfig
	}
	ServerConfig struct {
		Address string `config:"grpc-server-address,required"`
		Tvm     *tvm.TvmHelperConfig
	}
)

const (
	mockedAddress = ""
)

var (
	DefaultServerConfig = ServerConfig{
		Address: "127.0.0.1:9001",
		Tvm:     &tvm.TvmHelperConfig{EnabledServiceTicketCheck: false},
	}
	DefaultClientConfig = ClientConfig{
		Address: "127.0.0.1:9001",
		Tvm:     &clientTvmConfig{},
	}
	MockedServerConfig = ServerConfig{
		Address: mockedAddress,
		Tvm:     &tvm.TvmHelperConfig{EnabledServiceTicketCheck: false},
	}
	MockedClientConfig = ClientConfig{
		Address: mockedAddress,
		Tvm:     &clientTvmConfig{},
	}
	bufConnLis *bufconn.Listener
)

type ClientConnInterface interface {
	grpc.ClientConnInterface

	GetState() connectivity.State
	Close() error
}

func bufConnDialer(_ context.Context, _ string) (net.Conn, error) {
	return bufConnLis.Dial()
}

func InitBufConn() {
	bufConnLis = bufconn.Listen(1024 * 1024)
}

func CloseBufConn() {
	_ = bufConnLis.Close()
}

func NewServiceConnection(cfg *ClientConfig, logger *zap.Logger, ctx context.Context) (*grpc.ClientConn, error) {
	options := []grpc.DialOption{grpc.WithInsecure()}
	if cfg.Address == mockedAddress {
		if bufConnLis == nil {
			return nil, fmt.Errorf("NewServiceConnection: run InitBufConn() first")
		}
		options = append(options, grpc.WithContextDialer(bufConnDialer), grpc.WithDisableRetry())
	}
	if tvmCfg := cfg.Tvm; tvmCfg.ID != 0 {
		tvmHelper := tvm.NewDeployTvmHelper(logger, &tvm.TvmHelperConfig{SelfID: tvmCfg.SelfID})
		if tvmInterceptor := tvmHelper.GRPCClientInterceptor(tvmCfg.ID); tvmInterceptor != nil {
			options = append(options, grpc.WithUnaryInterceptor(tvmInterceptor))
		} else {
			return nil, fmt.Errorf("NewServiceConnection: cannot initialize tvm client")
		}
	}
	return grpc.DialContext(ctx, cfg.Address, options...)
}

type Server struct {
	cfg    *ServerConfig
	logger *zap.Logger
	server *grpc.Server
}

func NewServer(cfg *ServerConfig, metricsRegistry metrics.Registry, logger *zap.Logger) (*Server, error) {
	recoveryOptions := []recovery.Option{
		recovery.WithRecoveryHandler(func(p interface{}) error {
			err := status.Errorf(codes.Internal, "Panic in grpc: %v", p)
			logger.Error("Recovery in grpc", log.Error(err))
			return err
		}),
	}
	interceptors := []grpc.UnaryServerInterceptor{
		grpcOpentracing.UnaryServerInterceptor(),
		grpcLog.UnaryServerInterceptor(logger.L),
		recovery.UnaryServerInterceptor(recoveryOptions...),
		grpcMetrics.NewGrpcMetricsInterceptor(metricsRegistry),
	}
	tvmHelper := tvm.NewDeployTvmHelper(logger, cfg.Tvm)
	if tvmInterceptor := tvmHelper.CheckServiceTicketGRPCInterceptor(); tvmInterceptor != nil {
		interceptors = append(interceptors, tvmInterceptor)
	}
	server := grpc.NewServer(
		middleware.WithUnaryServerChain(interceptors...),
	)
	return &Server{
		cfg:    cfg,
		server: server,
		logger: logger,
	}, nil
}

func (s *Server) GetCoreServer() *grpc.Server {
	return s.server
}

func (s *Server) Serve() error {
	const logMessage = "Server:Serve"

	var lis net.Listener
	if s.cfg.Address == mockedAddress {
		if bufConnLis == nil {
			return fmt.Errorf("%s: run InitBufConn() first", logMessage)
		}
		lis = bufConnLis
	} else {
		var err error
		lis, err = net.Listen("tcp", s.cfg.Address)
		if err != nil {
			return fmt.Errorf("%s: failed to listen: %w", logMessage, err)
		}
	}
	s.logger.Infof("%s: start listening %s", logMessage, s.cfg.Address)
	err := s.server.Serve(lis)
	return err
}

func (s *Server) Stop() {
	s.server.Stop()
}
