package grpcgateway

import (
	"context"
	"net/http"
	"path"

	"github.com/go-chi/chi/v5"
	"github.com/go-chi/chi/v5/middleware"
	"github.com/grpc-ecosystem/grpc-gateway/runtime"

	"a.yandex-team.ru/library/go/core/resource"
	"a.yandex-team.ru/library/go/httputil/swaggerui"
)

type Config struct {
	Address string `config:"gateway-address,required"`
}

var DefaultConfig = Config{
	Address: "127.0.0.1:9002",
}

const swaggerPath = "/swagger/"

type Registrar interface {
	Register(ctx context.Context, endpoint string) (*runtime.ServeMux, error)
}

type Gateway struct {
	cfg             *Config
	swaggerJSONName string
	swaggerPrefix   string
	grpcAddr        string
	gatewayAddr     string
	registrar       Registrar
}

func NewGateway(cfg *Config, swaggerJSONName string, swaggerPrefix string,
	grpcAddr string, gatewayAddr string, registrar Registrar) *Gateway {
	return &Gateway{
		cfg:             cfg,
		swaggerJSONName: swaggerJSONName,
		swaggerPrefix:   swaggerPrefix,
		grpcAddr:        grpcAddr,
		gatewayAddr:     gatewayAddr,
		registrar:       registrar,
	}
}

func (g *Gateway) Run(ctx context.Context) error {
	swaggerRoot := path.Join(g.swaggerPrefix, swaggerPath)

	mux, err := g.registrar.Register(ctx, g.grpcAddr)
	if err != nil {
		return err
	}

	r := chi.NewRouter()
	r.Use(middleware.Logger)
	r.Handle("/*", mux)

	r.Route(swaggerRoot, func(root chi.Router) {
		fs := http.StripPrefix(
			swaggerRoot,
			http.FileServer(
				swaggerui.NewFileSystem(swaggerui.WithJSONScheme(
					resource.MustGet(g.swaggerJSONName),
				)),
			),
		)
		root.Get("/*", fs.ServeHTTP)
	})

	err = http.ListenAndServe(g.gatewayAddr, r)
	if err != nil {
		return err
	}
	return nil
}
