package logbroker

import (
	"os"
	"path"
	"time"

	"a.yandex-team.ru/kikimr/public/sdk/go/persqueue/recipe"
	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/library/go/core/log/zap"
	"a.yandex-team.ru/travel/library/go/logbroker"
	"a.yandex-team.ru/travel/library/go/vault"
	"a.yandex-team.ru/yt/go/yt"

	"a.yandex-team.ru/travel/buses/backend/internal/common/logging"
)

const (
	defaultEndpoint      = "logbroker.yandex.net"
	defaultLogLevel      = log.InfoLevel
	defaultReconnectWait = 15 * time.Second

	SearchResultTopic   = "search_result"
	SegmentsResultTopic = "segments_result"
)

type ProducerConfig struct {
	OAuthTokenSecret string `config:"logbroker-oauthtokensecret,required"`
	OAuthTokenKey    string `config:"logbroker-oauthtokenkey,required"`
	Endpoint         string `config:"logbroker-endpoint, required"`
	TopicPrefix      string `config:"logbroker-topicprefix, required"`
	SourceID         string `config:"logbroker-sourceid, required"`
	TestEnv          *recipe.Env
	LogLevel         log.Level
}

var DefaultProducerConfig = ProducerConfig{
	Endpoint: defaultEndpoint,
	SourceID: makeSourceID(),
	LogLevel: defaultLogLevel,
}

type ConsumerConfig struct {
	OAuthTokenSecret     string   `config:"logbrokerconsumer-oauthtokensecret,required"`
	OAuthTokenKey        string   `config:"logbrokerconsumer-oauthtokenkey,required"`
	Endpoint             string   `config:"logbrokerconsumer-endpoint, required"`
	Clusters             []string `config:"logbrokerconsumer-clusters, required"`
	Consumer             string   `config:"logbrokerconsumer-consumer, required"`
	TopicPrefix          string   `config:"logbrokerconsumer-topicprefix, required"`
	SourceID             string   `config:"logbrokerconsumer-sourceid, required"`
	TestEnv              *recipe.Env
	ReconnectWait        time.Duration `config:"logbroker-reconnectwait, required"`
	LogLevel             log.Level
	SearchYtLockPrefix   string `config:"logbrokerconsumer-searchytlockprefix, required"`
	SegmentsYtLockPrefix string `config:"logbrokerconsumer-segmentsytlockprefix, required"`
	YtLockPoolSize       uint   `config:"logbrokerconsumer-ytlockpoolsize, required"`
}

var DefaultDeployConsumerConfig = ConsumerConfig{
	Endpoint:      defaultEndpoint,
	Consumer:      "api",
	SourceID:      makeSourceID(),
	ReconnectWait: defaultReconnectWait,
	LogLevel:      defaultLogLevel,
}

func makeSourceID() string {
	return os.Getenv("DEPLOY_POD_ID")
}

func NewProducer(cfg ProducerConfig, topicSuffix string,
	secretResolver *vault.YavSecretsResolver, logger *zap.Logger) (*logbroker.Producer, error) {

	if cfg.TestEnv != nil {
		return logbroker.NewProducerWithRecipe(cfg.TestEnv, "", topicSuffix, logger), nil
	}

	topic := cfg.TopicPrefix + topicSuffix
	logger = logging.IncreaseLevel(logging.WithLogbrokerContext(logger, topic), cfg.LogLevel)
	return logbroker.NewProducer(
		cfg.TopicPrefix+topicSuffix,
		cfg.Endpoint,
		cfg.SourceID,
		logbroker.NewYavCredentialsProvider(
			cfg.OAuthTokenKey,
			cfg.OAuthTokenSecret,
			secretResolver),
		logger,
	)
}

func NewConsumer(cfg ConsumerConfig, topic string, ytLockPrefix string, readTimestamp time.Time,
	secretResolver *vault.YavSecretsResolver, ytClient yt.Client, logger *zap.Logger) (*logbroker.Consumer, error) {

	if cfg.TestEnv != nil {
		return logbroker.NewConsumerWithRecipe(cfg.TestEnv, []string{""}, topic, logger)
	}

	topic = path.Join(cfg.TopicPrefix, topic)
	lbConsumer := path.Join(cfg.TopicPrefix, cfg.Consumer)
	logger = logging.IncreaseLevel(logging.WithLogbrokerContext(logger, topic), cfg.LogLevel)

	return logbroker.NewConsumerWithLbConsumerGenerating(
		cfg.Clusters,
		cfg.Endpoint,
		topic,
		lbConsumer,
		readTimestamp,
		logbroker.NewYavCredentialsProvider(
			cfg.OAuthTokenKey,
			cfg.OAuthTokenSecret,
			secretResolver),
		logger,
		logbroker.NewYtLockGroup(ytClient, ytLockPrefix, cfg.YtLockPoolSize),
	)
}
