package utils

import (
	"fmt"
	"net/url"
	"strings"
	"time"

	"a.yandex-team.ru/travel/buses/backend/internal/common/dict"
	pbTravel "a.yandex-team.ru/travel/proto"
)

const (
	DateLayout         = "2006-01-02"
	BookURLTemplateStr = "//yandex.%s/bus/ride/%s/%s/%s/"
)

func LoadDate(date string) (*pbTravel.TDate, error) {
	t, err := time.Parse("2006-01-02", date)
	if err != nil {
		return &pbTravel.TDate{}, fmt.Errorf("bad date format: %w", err)
	}
	return ConvertTimeToProtoDate(t), nil
}

func DumpDate(date *pbTravel.TDate) string {
	return fmt.Sprintf("%d-%02d-%02d", date.Year, date.Month, date.Day)
}

func MakeRideID(supplier dict.Supplier, rideID string) string {
	return fmt.Sprintf("%s:%s", supplier.Name, rideID)
}

func ParseRideID(supplierRideID string) (uint32, string, error) {
	parts := strings.Split(supplierRideID, ":")
	if len(parts) != 2 || len(parts[0]) == 0 || len(parts[1]) == 0 {
		return 0, "", fmt.Errorf("ParseRideID: bad format")
	}
	supplier, err := dict.GetSupplierByName(parts[0])
	if err != nil {
		return 0, "", fmt.Errorf("ParseRideID: %w", err)
	}
	return supplier.ID, parts[1], nil
}

func BookURL(tld, fromStr, toStr, searchDate, rideID string, params url.Values) string {
	if tld == "" {
		tld = "ru"
	}
	prms := params.Encode()
	urlStr := fmt.Sprintf(BookURLTemplateStr, tld, fromStr, toStr, searchDate) + rideID
	if prms != "" {
		urlStr = urlStr + "?" + prms
	}
	return urlStr
}

func BookURLTemplate(tld, fromStr, toStr, searchDate string, params url.Values) string {
	if tld == "" {
		tld = "ru"
	}
	prms := params.Encode()
	urlStr := fmt.Sprintf(BookURLTemplateStr, tld, fromStr, toStr, searchDate) + "%s"
	if prms != "" {
		urlStr = urlStr + "?" + strings.Replace(prms, "%", "%%", -1)
	}
	return urlStr
}

func BookURLFromTemplate(template, rideID string) string {
	return fmt.Sprintf(template, rideID)
}
