package utils

import (
	"fmt"
	"strconv"
	"time"

	travelRasp "a.yandex-team.ru/travel/proto/dicts/rasp"

	"a.yandex-team.ru/travel/buses/backend/internal/common/dict/rasp"
	pb "a.yandex-team.ru/travel/buses/backend/proto"
)

const (
	typeStation    byte = 's'
	typeSettlement byte = 'c'
)

type PointKeyLoadError struct {
	PointKey string
}

func (pfe *PointKeyLoadError) Error() string {
	return fmt.Sprintf("pointkey load error for \"%s\"", pfe.PointKey)
}

type PointKeyDumpError struct {
	Type pb.EPointKeyType
	ID   uint32
}

func (pte *PointKeyDumpError) Error() string {
	return fmt.Sprintf("pointkey dump error for (%d, %d)", pte.Type, pte.ID)
}

func LoadPointKey(rawKey string) (*pb.TPointKey, error) {
	if len(rawKey) < 2 {
		return &pb.TPointKey{}, fmt.Errorf("too short: %w", &PointKeyLoadError{rawKey})
	}
	pointType := rawKey[0]
	pointID, err := strconv.ParseUint(rawKey[1:], 10, 64)
	if err != nil {
		return &pb.TPointKey{}, fmt.Errorf("bad id: %w", &PointKeyLoadError{rawKey})
	}
	var tp pb.EPointKeyType
	if pointType == typeStation {
		tp = pb.EPointKeyType_POINT_KEY_TYPE_STATION
	} else if pointType == typeSettlement {
		tp = pb.EPointKeyType_POINT_KEY_TYPE_SETTLEMENT
	} else {
		return &pb.TPointKey{}, fmt.Errorf("unknown type: %w", &PointKeyLoadError{rawKey})
	}

	return &pb.TPointKey{
		Type: tp,
		Id:   uint32(pointID),
	}, nil
}

func DumpPointKey(pk *pb.TPointKey) (string, error) {
	var tp byte
	if pk.Type == pb.EPointKeyType_POINT_KEY_TYPE_STATION {
		tp = typeStation
	} else if pk.Type == pb.EPointKeyType_POINT_KEY_TYPE_SETTLEMENT {
		tp = typeSettlement
	} else {
		return "", fmt.Errorf("unknown type: %w", &PointKeyDumpError{pk.Type, pk.Id})
	}
	return fmt.Sprintf("%c%d", tp, pk.Id), nil
}

func GetPointKeyTimeZone(raspRepo *rasp.DictRepo, pk *pb.TPointKey) (*time.Location, error) {
	const logMessage = "GetPointKeyTimeZone"

	if pk == nil {
		return nil, fmt.Errorf("%s: PointKey is nil", logMessage)
	}

	var tZ *travelRasp.TTimeZone
	if pk.Type == pb.EPointKeyType_POINT_KEY_TYPE_SETTLEMENT {
		settlement, ok := raspRepo.GetSettlement(int32(pk.Id))
		if !ok {
			return nil, fmt.Errorf("%s: Settlement=%d not found in RaspRepo", logMessage, pk.Id)
		}
		tZ, ok = raspRepo.GetTimeZone(settlement.TimeZoneId)
		if !ok {
			return nil, fmt.Errorf("%s: TimeZone=%d not found in RaspRepo", logMessage, settlement.TimeZoneId)
		}
	} else {
		if pk.Type == pb.EPointKeyType_POINT_KEY_TYPE_STATION {
			station, ok := raspRepo.GetStation(int32(pk.Id))
			if !ok {
				return nil, fmt.Errorf("%s: Station=%d not found in RaspRepo", logMessage, pk.Id)
			}
			tZ, ok = raspRepo.GetTimeZone(station.TimeZoneId)
			if !ok {
				return nil, fmt.Errorf("%s: TimeZone=%d not found in RaspRepo", logMessage, station.TimeZoneId)
			}
		} else {
			return nil, fmt.Errorf("%s: unknown PointKey type=%s", logMessage, pk.Type)
		}
	}
	location, err := time.LoadLocation(tZ.Code)
	if err != nil {
		return nil, fmt.Errorf("%s: %w", logMessage, err)
	}
	return location, nil
}
