package app

import (
	"context"
	"errors"
	"fmt"
	"sync"

	"a.yandex-team.ru/library/go/core/log/zap"
	"a.yandex-team.ru/library/go/core/metrics"
	"a.yandex-team.ru/travel/library/go/logbroker"
	travelMetrics "a.yandex-team.ru/travel/library/go/metrics"
	"a.yandex-team.ru/travel/library/go/unifiedagent"
	"a.yandex-team.ru/travel/library/go/vault"
	tpb "a.yandex-team.ru/travel/proto"
	"google.golang.org/grpc"

	"a.yandex-team.ru/travel/buses/backend/internal/common/connector"
	commonGRPC "a.yandex-team.ru/travel/buses/backend/internal/common/grpc"
	ilogbroker "a.yandex-team.ru/travel/buses/backend/internal/common/logbroker"
	"a.yandex-team.ru/travel/buses/backend/internal/common/logging"
	workerLogging "a.yandex-team.ru/travel/buses/backend/internal/worker/logging"
	"a.yandex-team.ru/travel/buses/backend/internal/worker/task"
	tcpb "a.yandex-team.ru/travel/buses/backend/proto/testcontext"
	wpb "a.yandex-team.ru/travel/buses/backend/proto/worker"
)

type Config struct {
	Connector          connector.Config
	Logbroker          ilogbroker.ProducerConfig
	TestContext        commonGRPC.ClientConfig
	UnifiedAgentClient unifiedagent.ClientConfig
}

var DefaultConfig = Config{
	Connector:          connector.DefaultConfig,
	Logbroker:          ilogbroker.DefaultProducerConfig,
	TestContext:        commonGRPC.DefaultClientConfig,
	UnifiedAgentClient: unifiedagent.DefaultClientConfig,
}

const (
	moduleName = "app"
)

type App struct {
	wpb.UnimplementedWorkerServiceServer

	logger    *zap.Logger
	ctx       context.Context
	ctxCancel context.CancelFunc
	cfg       *Config

	appMetrics *travelMetrics.AppMetrics

	testContextServiceConnection *grpc.ClientConn
	testContextServiceClient     tcpb.BusesTestContextServiceClient

	searchTaskQueue  *task.SearchTaskQueue
	searchWorker     *task.SearchWorker
	searchProducer   *logbroker.Producer
	segmentsProducer *logbroker.Producer

	activeSegmentsRequests      map[uint32]struct{}
	activeSegmentsRequestsMutex sync.RWMutex

	communicationLogger *workerLogging.CommunicationLogWriter
}

func NewApp(cfg *Config, metricsRegistry metrics.Registry, logger *zap.Logger) (*App, error) {
	const logMessage = "App.NewApp"
	moduleLogger := logging.WithModuleContext(logger, moduleName)

	appMetrics := travelMetrics.NewAppMetrics(metricsRegistry)

	searchTaskQueue := task.NewSearchTaskQueue(appMetrics)

	secretResolver := vault.NewYavSecretsResolver()
	searchProducer, err := ilogbroker.NewProducer(
		cfg.Logbroker, ilogbroker.SearchResultTopic, secretResolver, logger,
	)
	if err != nil {
		return nil, fmt.Errorf("%s: can not create search producer: %w", logMessage, err)
	}
	segmentsProducer, err := ilogbroker.NewProducer(
		cfg.Logbroker, ilogbroker.SegmentsResultTopic, secretResolver, logger,
	)
	if err != nil {
		return nil, fmt.Errorf("%s: can not create segments producer: %w", logMessage, err)
	}

	ctx, ctxCancel := context.WithCancel(context.Background())

	communicationLogger, err := workerLogging.NewCommunicationLogger(&cfg.UnifiedAgentClient, logger)
	if err != nil {
		ctxCancel()
		return nil, fmt.Errorf("failed to create communication logger: %s", err)
	}
	testContextServiceConnection, err := commonGRPC.NewServiceConnection(&cfg.TestContext, logger, ctx)
	if err != nil {
		ctxCancel()
		return nil, fmt.Errorf("can not connect to testcontext service at %s: %w", cfg.TestContext.Address, err)
	}

	testContextServiceClient := tcpb.NewBusesTestContextServiceClient(testContextServiceConnection)

	return &App{
		logger:    moduleLogger,
		ctx:       ctx,
		ctxCancel: ctxCancel,
		cfg:       cfg,

		appMetrics: appMetrics,

		testContextServiceConnection: testContextServiceConnection,
		testContextServiceClient:     testContextServiceClient,

		searchTaskQueue: searchTaskQueue,
		searchWorker: task.NewSearchWorker(
			searchTaskQueue,
			&cfg.Connector,
			searchProducer,
			logger,
			communicationLogger,
			metricsRegistry,
		),
		searchProducer:   searchProducer,
		segmentsProducer: segmentsProducer,

		activeSegmentsRequests: make(map[uint32]struct{}),
		communicationLogger:    communicationLogger,
	}, nil
}

func (a *App) Run() error {
	const logMessage = "App.Run"
	if err := a.searchProducer.Run(a.ctx); err != nil {
		return fmt.Errorf("%s: failed to run search producer: %w", logMessage, err)
	}
	if err := a.segmentsProducer.Run(a.ctx); err != nil {
		return fmt.Errorf("%s: failed to run segments producer: %w", logMessage, err)
	}
	a.searchWorker.Run(a.ctx)
	return nil
}

func (a *App) Close() {
	a.ctxCancel()
	_ = a.searchProducer.Close()
	_ = a.segmentsProducer.Close()
	_ = a.communicationLogger.Close()
	_ = a.testContextServiceConnection.Close()
}

func headerFromError(err error) *wpb.TResponseHeader {
	var (
		errorProto       tpb.TError
		connectorError   connector.ErrWithMetadata
		unavailableError connector.ErrUnavailable
	)

	if errors.As(err, &connectorError) {
		errorProto.Code = tpb.EErrorCode_EC_ABORTED
		errorProto.Message = connectorError.Error()
	} else if errors.As(err, &unavailableError) {
		errorProto.Code = tpb.EErrorCode_EC_UNAVAILABLE
		errorProto.Message = unavailableError.Error()
	} else {
		errorProto.Code = tpb.EErrorCode_EC_GENERAL_ERROR
		errorProto.Message = err.Error()
	}

	return &wpb.TResponseHeader{Code: errorProto.Code, Error: &errorProto}
}
