package app

import (
	"context"
	"fmt"
	"time"

	tpb "a.yandex-team.ru/travel/proto"

	"google.golang.org/protobuf/proto"

	"a.yandex-team.ru/travel/buses/backend/internal/common/connector"
	"a.yandex-team.ru/travel/buses/backend/internal/common/dict"
	wpb "a.yandex-team.ru/travel/buses/backend/proto/worker"
)

func (a *App) Segments(_ context.Context, request *wpb.TSegmentsRequest) (*wpb.TSegmentsResponse, error) {

	timestamp := time.Now().Unix()
	supplierID := request.SupplierId

	if _, err := dict.GetSupplier(supplierID); err != nil {
		return &wpb.TSegmentsResponse{
			Header: &wpb.TResponseHeader{
				Code: tpb.EErrorCode_EC_GENERAL_ERROR,
				Error: &tpb.TError{
					Code:    tpb.EErrorCode_EC_GENERAL_ERROR,
					Message: err.Error(),
				},
				Timestamp: timestamp,
			},
		}, nil
	}

	go a.getSegments(request)

	return &wpb.TSegmentsResponse{
		Header: &wpb.TResponseHeader{
			Code:      tpb.EErrorCode_EC_OK,
			Timestamp: timestamp,
		},
	}, nil
}

func (a *App) getSegments(request *wpb.TSegmentsRequest) {
	const logMessage = "App.getSegments"

	a.activeSegmentsRequestsMutex.Lock()
	_, ok := a.activeSegmentsRequests[request.SupplierId]
	if ok {
		a.activeSegmentsRequestsMutex.Unlock()
		return
	}
	a.activeSegmentsRequests[request.SupplierId] = struct{}{}
	a.activeSegmentsRequestsMutex.Unlock()

	result := &wpb.TSegmentsResult{
		Request: request,
	}

	defer func() {
		resultSegments := proto.Clone(result).(*wpb.TSegmentsResult)
		resultSegments.Header = nil
		resultSegments.Request = nil
		a.communicationLogger.SendLog(wpb.ECommunicationLogRecordType_LRT_WORKER_SEGMENTS, request.GetHeader(),
			request, result.GetHeader(), resultSegments)
	}()

	connectorClient, err := connector.NewClient(&a.cfg.Connector, request.SupplierId, a.logger)
	if err != nil {
		a.logger.Errorf("%s: %s", logMessage, err.Error())
		return
	}

	segments, explanation, err := connectorClient.GetSegments()
	timestamp := time.Now().Unix()
	if err != nil {
		msg := fmt.Sprintf("%s: error getting segments from partner %d: %s", logMessage, request.SupplierId, err.Error())
		a.logger.Error(msg)
		result.Header = &wpb.TResponseHeader{
			Code: tpb.EErrorCode_EC_GENERAL_ERROR,
			Error: &tpb.TError{
				Code:    tpb.EErrorCode_EC_GENERAL_ERROR,
				Message: msg,
			},
			Timestamp: timestamp,
		}
	} else {
		result.Header = &wpb.TResponseHeader{
			Code:      tpb.EErrorCode_EC_OK,
			Timestamp: timestamp,
		}
		result.Segments = segments
	}
	result.Header.Explanation = explanation

	err = a.segmentsProducer.Write(result)
	if err != nil {
		a.logger.Errorf("%s: can not write to segments producer: %s", logMessage, err.Error())
	}

	a.activeSegmentsRequestsMutex.Lock()
	delete(a.activeSegmentsRequests, request.SupplierId)
	a.activeSegmentsRequestsMutex.Unlock()
}
