package logging

import (
	"fmt"

	"a.yandex-team.ru/library/go/core/log/zap"
	"a.yandex-team.ru/travel/buses/backend/internal/common/logging"
	"a.yandex-team.ru/travel/library/go/unifiedagent"
	"google.golang.org/protobuf/encoding/protojson"
	"google.golang.org/protobuf/proto"

	wpb "a.yandex-team.ru/travel/buses/backend/proto/worker"
)

const (
	moduleName = "communication"
)

type CommunicationLogWriter struct {
	unifiedAgent unifiedagent.Client
	logger       *zap.Logger
}

func NewCommunicationLogger(cfg *unifiedagent.ClientConfig, logger *zap.Logger) (*CommunicationLogWriter, error) {
	const funcName = "NewCommunicationLogger"
	moduleLogger := logging.WithModuleContext(logger, moduleName)

	unifiedAgentClient, err := unifiedagent.NewGrpcClient(cfg, moduleLogger, nil)
	if err != nil {
		return nil, fmt.Errorf(
			"%s: can not create unifiedAgentClient at %s: %w",
			funcName, cfg.Endpoint, err,
		)
	}

	return &CommunicationLogWriter{
		unifiedAgent: unifiedAgentClient,
		logger:       moduleLogger,
	}, nil
}

func (cl *CommunicationLogWriter) SendLog(
	logType wpb.ECommunicationLogRecordType, requestHeader *wpb.TRequestHeader, request proto.Message,
	responseHeader *wpb.TResponseHeader, responsePayload proto.Message) {
	const funcName = "SendLog"

	var (
		err          error
		requestJSON  []byte
		responseJSON []byte
	)
	if request != nil {
		requestJSON, err = protojson.Marshal(request)
		if err != nil {
			cl.logger.Errorf("%s: %s", funcName, err.Error())
			return
		}
	}
	if responsePayload != nil {
		responseJSON, err = protojson.Marshal(responsePayload)
		if err != nil {
			cl.logger.Errorf("%s: %s", funcName, err.Error())
			return
		}
	}
	logRecord := &wpb.TCommunicationLogRecord{
		Type:             logType,
		TestContextToken: requestHeader.GetTestContextToken(),
		RequestId:        requestHeader.GetRequestId(),
		Request:          requestJSON,
		Response:         responseJSON,
		Code:             responseHeader.GetCode(),
		Error:            responseHeader.GetError(),
		Timestamp:        responseHeader.GetTimestamp(),
		Explanation:      responseHeader.GetExplanation(),
	}
	logRecordCopy := proto.Clone(logRecord)
	go func() {
		err = cl.unifiedAgent.Send(logRecordCopy, nil)
		if err != nil {
			cl.logger.Errorf("%s: %s", funcName, err.Error())
		}
	}()
}

func (cl *CommunicationLogWriter) Close() error {
	return cl.unifiedAgent.Close()
}
