package task

import (
	"container/list"
	"errors"
	"fmt"
	"sync"

	"github.com/golang/protobuf/proto"

	coreMetrics "a.yandex-team.ru/library/go/core/metrics"
	"a.yandex-team.ru/travel/buses/backend/internal/api/cache"
	"a.yandex-team.ru/travel/buses/backend/internal/common/dict"
	wpb "a.yandex-team.ru/travel/buses/backend/proto/worker"
	"a.yandex-team.ru/travel/library/go/metrics"
)

var supportedPriorities = []wpb.ERequestPriority{
	wpb.ERequestPriority_REQUEST_PRIORITY_HIGH,
	wpb.ERequestPriority_REQUEST_PRIORITY_NORMAL,
	wpb.ERequestPriority_REQUEST_PRIORITY_LOW,
}
var ErrNoTasks = errors.New("no more tasks")

// NOTE: PriorityQueue is not thread safe
type PriorityQueue struct {
	queues      map[wpb.ERequestPriority]*list.List
	metricsLen  map[wpb.ERequestPriority]coreMetrics.Gauge
	metricsPush map[wpb.ERequestPriority]coreMetrics.Counter
	metricsPop  map[wpb.ERequestPriority]coreMetrics.Counter
}

func NewPriorityQueue(appMetrics *metrics.AppMetrics, metricsTagKey string, metricsTagValue string) *PriorityQueue {
	q := PriorityQueue{
		queues:      make(map[wpb.ERequestPriority]*list.List),
		metricsLen:  make(map[wpb.ERequestPriority]coreMetrics.Gauge),
		metricsPush: make(map[wpb.ERequestPriority]coreMetrics.Counter),
		metricsPop:  make(map[wpb.ERequestPriority]coreMetrics.Counter),
	}
	for _, priority := range supportedPriorities {
		q.queues[priority] = list.New()
		q.metricsLen[priority] = appMetrics.GetOrCreateGauge(
			"task_queue",
			map[string]string{
				metricsTagKey: metricsTagValue,
				"priority":    priority.String(),
			},
			"len")
		q.metricsPush[priority] = appMetrics.GetOrCreateCounter(
			"task_queue",
			map[string]string{
				metricsTagKey: metricsTagValue,
				"priority":    priority.String(),
			},
			"push_count")
		q.metricsPop[priority] = appMetrics.GetOrCreateCounter(
			"task_queue",
			map[string]string{
				metricsTagKey: metricsTagValue,
				"priority":    priority.String(),
			},
			"pop_count")
	}
	return &q
}

// Notes
// HIGH:   sync requests
// NORMAL: user generated
// LOW:    warmer generated
func (q *PriorityQueue) Push(task proto.Message, priority wpb.ERequestPriority) (int, error) {
	queue, ok := q.queues[priority]
	if !ok {
		return -1, fmt.Errorf("PriorityQueue.Push: unknown priority=%s", priority)
	}
	queue.PushBack(task)
	queueLen := queue.Len()
	metricLen := q.metricsLen[priority]
	metricLen.Set(float64(queueLen))
	metricPush := q.metricsPush[priority]
	metricPush.Inc()
	return queueLen, nil
}

func (q *PriorityQueue) Pop() (proto.Message, error) {
	for _, priority := range supportedPriorities {
		queue, ok := q.queues[priority]
		if !ok {
			return nil, fmt.Errorf("PriorityQueue.Pop: unknown priority=%s", priority)
		}
		if queue.Len() > 0 {
			element := queue.Front()
			queue.Remove(element)
			metricLen := q.metricsLen[priority]
			metricLen.Set(float64(queue.Len()))
			metricPop := q.metricsPop[priority]
			metricPop.Inc()
			return element.Value.(proto.Message), nil
		}
	}
	return nil, ErrNoTasks
}

func (q *PriorityQueue) GetList(priority wpb.ERequestPriority) (*list.List, error) {
	queue, ok := q.queues[priority]
	if !ok {
		return nil, fmt.Errorf("PriorityQueue.GetList: unknown priority=%s", priority)
	}
	return queue, nil
}

func (q *PriorityQueue) Len() int {
	len := 0
	for _, queue := range q.queues {
		len += queue.Len()
	}
	return len
}

type SearchTaskQueue struct {
	supplierPriorityQueue map[uint32]*PriorityQueue
	index                 map[wpb.ERequestPriority]map[cache.SearchKey]int
	mutex                 sync.Mutex
	appMetrics            *metrics.AppMetrics
}

func NewSearchTaskQueue(appMetrics *metrics.AppMetrics) *SearchTaskQueue {
	sq := SearchTaskQueue{
		supplierPriorityQueue: make(map[uint32]*PriorityQueue),
		index:                 make(map[wpb.ERequestPriority]map[cache.SearchKey]int),
		appMetrics:            appMetrics,
	}
	for _, suppierID := range dict.GetSuppliersList() {
		supplier, _ := dict.GetSupplier(suppierID)
		sq.supplierPriorityQueue[suppierID] = NewPriorityQueue(appMetrics, "supplier", supplier.Name)
	}
	for _, priority := range supportedPriorities {
		sq.index[priority] = make(map[cache.SearchKey]int)
	}
	return &sq
}

func (sq *SearchTaskQueue) Push(request *wpb.TSearchRequest) (int, error) {
	if request.From == nil || request.To == nil || request.Date == nil || request.Header == nil {
		return -1, fmt.Errorf("SearchTaskQueue.Push: bad request")
	}

	sq.mutex.Lock()
	defer sq.mutex.Unlock()

	searchKey := cache.NewSearchKey(request.SupplierId, request.From, request.To, request.Date)
Loop:
	for _, priority := range supportedPriorities {
		priorityIndex := sq.index[priority]
		i, ok := priorityIndex[searchKey]
		if !ok {
			continue
		}
		if priority >= request.Header.Priority {
			return i, nil
		}
		delete(priorityIndex, searchKey)
		priorityQueue, err := sq.getSupplierQueue(request.SupplierId)
		if err != nil {
			return -1, err
		}
		priorityList, err := priorityQueue.GetList(priority)
		if err != nil {
			return -1, err
		}
		for element := priorityList.Front(); element != nil; element = element.Next() {
			requestInList := element.Value.(*wpb.TSearchRequest)
			if searchKey == cache.NewSearchKey(requestInList.SupplierId, requestInList.From,
				requestInList.To, requestInList.Date) {
				priorityList.Remove(element)
				break Loop
			}
		}
		return -1, fmt.Errorf("SearchTaskQueue.Push: internal error, index and data are not synced")
	}
	priorityQueue, err := sq.getSupplierQueue(request.SupplierId)
	if err != nil {
		return -1, err
	}
	i, err := priorityQueue.Push(request, request.Header.Priority)
	if err != nil {
		return -1, err
	}
	priorityIndex := sq.index[request.Header.Priority]
	priorityIndex[searchKey] = i
	return i, nil
}

func (sq *SearchTaskQueue) Pop(supplierID uint32) (*wpb.TSearchRequest, error) {
	sq.mutex.Lock()
	defer sq.mutex.Unlock()

	priorityQueue, err := sq.getSupplierQueue(supplierID)
	if err != nil {
		return nil, err
	}
	message, err := priorityQueue.Pop()
	if err != nil {
		return nil, err
	}
	request := message.(*wpb.TSearchRequest)
	searchKey := cache.NewSearchKey(request.SupplierId, request.From, request.To, request.Date)
	for _, priority := range supportedPriorities {
		priorityIndex := sq.index[priority]
		delete(priorityIndex, searchKey)
	}
	return request, nil
}

func (sq *SearchTaskQueue) Len(supplierID uint32) (int, error) {
	sq.mutex.Lock()
	defer sq.mutex.Unlock()

	priorityQueue, err := sq.getSupplierQueue(supplierID)
	if err != nil {
		return 0, err
	}
	return priorityQueue.Len(), nil
}

func (sq *SearchTaskQueue) getSupplierQueue(supplierID uint32) (*PriorityQueue, error) {
	priorityQueue, ok := sq.supplierPriorityQueue[supplierID]
	if !ok {
		return nil, fmt.Errorf("SearchTaskQueue.getSupplierQueue: unknown SupplierID=%d", supplierID)
	}
	return priorityQueue, nil
}
