# coding: utf-8
from __future__ import absolute_import, division, print_function, unicode_literals

import logging
import os
from datetime import timedelta
from os import getenv

from six import text_type

MONITORING = {
    'host': '::',
    'port': 9947,
    'endpoint': '/metrics/',
    'common_labels': {
        'project': 'buses',
        'service': 'connectors',
    }
}

RESOURCE_TIMEOUT = 140  # seconds
SLOW_RESOURCE_TIMEOUT = 300  # seconds
SLOW_RESOURCES = frozenset(map(text_type.strip, getenv('SLOW_RESOURCES', '').split(',')))  # opts: endpoints, segments
SLOW_CONNECTORS = frozenset(map(text_type.strip, getenv('SLOW_CONNECTORS', '').split(',')))

REQUEST_READ_TIMEOUT = 140  # seconds
REQUEST_LONG_READ_TIMEOUT = 300  # seconds

CONVERTER_UPDATE_PERIOD = 30  # minutes

ERROR_404_HELP = False
DEBUG = True
PROPAGATE_EXCEPTIONS = False
RESTFUL_JSON = {'ensure_ascii': False, 'encoding': 'utf-8'}

MAX_POOL_SIZE = int(getenv('MAX_POOL_SIZE', 1000))

RASP_DATA_PATH = getenv('RASP_DATA_PATH', None) or '/var/lib/resources/rasp_data'

BUSFOR_FEES_REVENUE_RATE = 0.67  # revenue rate of busfor fees

ETRAFFIC_SEGMENTS_UPDATE_PERIOD = timedelta(days=1)
ETRAFFIC_SEGMENTS_DUMP_PATH = '/ephemeral/etraffic_segments.json'

YUGAVTOTRANS_CARRIERS_UPDATE_PERIOD = timedelta(hours=24)


class Log(object):
    LEVEL = 'INFO'
    CONNECTOR_RESPONSE_LEVEL = 'INFO'
    DIRECTORY = '/var/log/app/'

    ERROR_BOOSTER_ENABLED = True
    ERROR_BOOSTER_BREADCRUMBS_LEVEL = logging.INFO
    ERROR_BOOSTER_EVENT_LEVEL = logging.ERROR

    @classmethod
    def get_config_dict(cls):
        return {
            'version': 1,
            'disable_existing_loggers': False,
            'filters': {
                'connector_order_methods': {
                    '()': 'yabus.util.logging.GenericFilter',
                    'predicate': lambda record:
                        getattr(record, 'method_name', None) in ('book', 'confirm', 'refund_info', 'refund', 'cancel')
                }
            },
            'formatters': {
                'default': {
                    'format': '%(levelname)s %(asctime)s %(name)s %(message)s'
                },
                'environment_json': {
                    '()': 'ylog.QloudJsonFormatter'  # it has Deploy support despite naming
                },
                'json': {
                    '()': 'pythonjsonlogger.jsonlogger.JsonFormatter',
                    'format': '%(message)s',
                },
            },
            'handlers': {
                'application': {
                    'class': 'logging.handlers.WatchedFileHandler',
                    'filename': os.path.join(cls.DIRECTORY, 'application.log'),
                    'formatter': 'default',
                },
                'stdout_application': {
                    'class': 'logging.StreamHandler',
                    'stream': 'ext://sys.stdout',
                    'formatter': 'environment_json'
                },
                'zeep': {
                    'class': 'logging.handlers.WatchedFileHandler',
                    'filename': os.path.join(cls.DIRECTORY, 'zeep.log'),
                    'formatter': 'default',
                },
                'connector_response': {
                    'class': 'logging.handlers.WatchedFileHandler',
                    'filename': os.path.join(cls.DIRECTORY, 'connector_response.log'),
                    'formatter': 'default',
                    'level': cls.CONNECTOR_RESPONSE_LEVEL,
                },
                'connector_order_response': {
                    'class': 'logging.handlers.WatchedFileHandler',
                    'filename': os.path.join(cls.DIRECTORY, 'yt', 'order.log'),
                    'filters': ['connector_order_methods'],
                    'formatter': 'default',
                    'level': 'DEBUG',
                },
            },
            'loggers': {
                '': {
                    'handlers': ['application', 'stdout_application'],
                    'level': cls.LEVEL,
                },
                'zeep.transports': {
                    'handlers': ['zeep'],
                    'level': cls.LEVEL,
                    'propagate': False,
                },
                'yabus.util.connector_response': {
                    'handlers': ['connector_response', 'connector_order_response'],
                    'level': 'DEBUG',  # actual level is defined by handlers instead of parent loggers
                    'propagate': False,
                },
            }
        }
